// maya.cmd  -  using one mesh for whole surface

/* Surface Evolver script to write Maya ASCII file.
   Reference: http://caad.arch.ethz.ch/info/maya/manual/FileFormats/index.html
   Also in Maya documentation:
      Developer Resources > File Formats > Maya ASCII file format > Maya ASCII file format >  
          Organizationof Maya Ascii files
   Usage:
    Enter command: read "maya.cmd"
    Enter command: maya >>> "filename.ma"

   Note: The default length unit is cm, but if you want to use something
   different, then set the maya_length_unit variable to the appropriate string
   after loading maya.cmd and before running the maya command,
   e.g.

    Enter command: maya_length_unit := "in"
    Enter command: maya >>> "filename.ma"

   Valid length unit strings are:
    "mm", "millimeter", "cm", "centimeter", "m", "meter", "km", "kilometer", 
    "in", "inch", "ft", "foot", "yd", "yard", "mi", "mile".

   Note: To prevent names generated by this script from conflicting with existing
     maya names or names from another datafile, there is a string variable 
     maya_name that is prepended to all names generated by this script.
     There is a maya namespace feature that probably could be used in place
     of this, but I do not understand it yet.  The maya_name default is "AAA";
     I didn't use the datafilename since this may contain illegal characters
     and would prevent generating multiple maya objects from the same datafile.
     So if you are going to use multiple surfaces simultaneously, change
     maya_name before executing the maya command.
*/

maya_length_unit := "cm"
maya_name := "AAA"

// Evolver's RGB color components; index is color+1
define rgb_maya integer[16][3];
rgb_maya[1][1] := 0; rgb_maya[1][2] := 0; rgb_maya[1][3] := 0;
rgb_maya[2][1] := 0; rgb_maya[2][2] := 0; rgb_maya[2][3] := 255;
rgb_maya[3][1] := 0; rgb_maya[3][2] := 255; rgb_maya[3][3] := 0;
rgb_maya[4][1] := 0; rgb_maya[4][2] := 255; rgb_maya[4][3] := 255;
rgb_maya[5][1] := 255; rgb_maya[5][2] := 0; rgb_maya[5][3] := 0;
rgb_maya[6][1] := 255; rgb_maya[6][2] := 0; rgb_maya[6][3] := 255;
rgb_maya[7][1] := 255; rgb_maya[7][2] := 127; rgb_maya[7][3] := 0;
rgb_maya[8][1] := 160; rgb_maya[8][2] := 160; rgb_maya[8][3] := 160;
rgb_maya[9][1] := 80; rgb_maya[9][2] := 80; rgb_maya[9][3] := 80;
rgb_maya[10][1] := 80; rgb_maya[10][2] := 200; rgb_maya[10][3] := 255;
rgb_maya[11][1] := 127; rgb_maya[11][2] := 255; rgb_maya[11][3] := 127;
rgb_maya[12][1] := 127; rgb_maya[12][2] := 255; rgb_maya[12][3] := 255;
rgb_maya[13][1] := 255; rgb_maya[13][2] := 127; rgb_maya[13][3] := 127;
rgb_maya[14][1] := 255; rgb_maya[13][2] := 127; rgb_maya[13][3] := 255;
rgb_maya[15][1] := 255; rgb_maya[15][2] := 255; rgb_maya[15][3] := 0;
rgb_maya[16][1] := 255; rgb_maya[16][2] := 255; rgb_maya[16][3] := 255;

// Need consecutive numbering; Evolver element id numbers may not be.
define vertex attribute mayavinx integer
define edge attribute mayaeinx integer

maya := {
  local ecount,fcount,inx,vertexinx,edgeinx;

  // Check assumptions.

 if torus then
  { errprintf "Cannot run 'maya' command in torus mode. Do 'detorus' first.\n";
    errprintf "    WARNING: 'detorus' will modify the surface; save it!\n";
    abort;
  };

  if symmetry_group then
  { errprintf "Cannot run 'maya' command in symmetry group mode. Do 'detorus' first.\n";
    errprintf "    WARNING: 'detorus' will modify the surface; save it!\n";
    abort;
  };

  if space_dimension != 3 then
  { errprintf "The 'maya' command must be run in three-dimensional space.\n";
    abort;
  };


  if surface_dimension == 1 then
  { errprintf "The 'maya' command is not meant for the string model.\n";
    abort;
  };

  if simplex_representation then
  { errprintf "The 'maya' command is not meant for the simplex model.\n";
    abort;
  };

  if lagrange_order >= 2 then
  { errprintf "The 'maya' command is meant for the linear model, not quadratic or Lagrange.\n";
    abort;
  };

  // Header section.  Just comments.
  printf "//Maya ASCII 2009 scene\n";   // required first six bytes
  printf "// Maya file created from Surface Evolver datafile %s\n",datafilename;
  printf "// Date: %s\n",date_and_time;

  // File reference section. None at present.

  // Requirements section.  No special requirements that I know of.
  printf "requires maya \"2009\";\n";

  // Units section.  User might have to edit this, since Evolver doesn't
  // know.  Using centimeter as default length unit.
  printf "currentUnit -l \"%s\";\n",maya_length_unit;

  // File information section.  Try some home-grown ones.
  printf "fileInfo \"datafile\"  \"%s\";\n",datafilename;
  printf "fileInfo \"generator program\"  \"Surface Evolver\";\n";
  printf "fileInfo \"generator script\"  \"maya.cmd\";\n";

  // Main section.  Nodes, attributes, and parenting.

  // One transform node for the whole surface
  printf "createNode transform -n \"%sSurface\";\n",maya_name;

  // One mesh node per facet.  Should be a way to use one mesh node
  // for bunch of facets, but I haven't been able to generate an
  // example yet.
  printf "createNode mesh -n \"%sShape\" -p \"%sSurface\";\n",maya_name,maya_name;
  printf "    setAttr -k off \".v\";\n";
  printf "    setAttr -s 2 \".iog[0].og\";\n";
  printf "    setAttr \".iog[0].og[1].gcl\" -type \"componentList\" 1 \"f[0:%d]\";\n",
      facet_count-1;
  printf "    setAttr \".uvst[0].uvsn\" -type \"string\" \"map1\";\n";
  printf "    setAttr \".cuvs\" -type \"string\" \"map1\";\n";
  printf "    setAttr \".dcc\" -type \"string\" \"Ambient+Diffuse\";\n";
  printf "    setAttr -s 16 \".clr\";\n";
  printf "    setAttr \".clr[0:15]\"\n";
  for ( inx := 1 ; inx <= 16 ; inx += 1 )
    printf "       %f %f %f 0.0\n",rgb_maya[inx][1]/255,rgb_maya[inx][2]/255,rgb_maya[inx][3]/255;
  printf "       ;\n";
  printf "    setAttr -s %d \".vt\";\n",vertex_count;
  printf "    setAttr \".vt[0:%d]\" \n",vertex_count-1;
  vertexinx := 0;
  foreach vertex vv do
  { printf "        %f %f %f\n",vv.y,vv.z,vv.x;
    vv.mayavinx := vertexinx;
    vertexinx += 1;
  };
  printf "       ;\n";
  ecount := sum(edge,show);
  printf "    setAttr -s %d \".ed\";\n",ecount;
  printf "    setAttr \".ed[0:%d]\"\n",ecount-1;
  edgeinx := 0;
  foreach edge ee where show and color >= 0 do 
  { printf "     %d %d 0\n",ee.vertex[1].mayavinx,ee.vertex[2].mayavinx;
    ee.mayaeinx := edgeinx;
    edgeinx += 1;
  };
  printf "       ;\n";
  fcount := sum(facet,show);
  printf "    setAttr -s %d \".fc\";\n",fcount;
  printf "    setAttr \".fc[0:%d]\" -type \"polyFaces\"\n",fcount-1;
  foreach facet ff where show and color >= 0  do
  { printf "        f 3 %d %d %d\n",ff.edge[1].mayaeinx,ff.edge[2].mayaeinx,ff.edge[3].mayaeinx;
    printf "        fc 1 %d\n",ff.color;
  };
  printf "       ;\n";

  // Create a material 
  printf "createNode materialInfo -n \"%smaterial\";\n",maya_name;
  printf "createNode shadingEngine -n \"%slambertSG\";\n",maya_name;
  printf "    setAttr \".ihi\" 0;\n";
  printf "    setAttr \".ro\" yes;\n";
  printf "createNode lambert -n \"%slambert\";\n",maya_name;

  // And this node, just because it's in the examples I generated
  printf "\n";
  printf "createNode lightLinker -n \"%slightLinker\";\n",maya_name;
  printf "    setAttr -s %d \".lnk\";\n",facet_count;
  printf "    setAttr -s %d \".slnk\";\n",facet_count;

  // Stuff I have little clue about
  printf "\n";
  printf "select -ne :renderPartition;\n";
  printf "    setAttr -s 4 \".st\";\n"; 
  printf "\n";
  printf "select -ne :defaultShaderList1;\n";
  printf "    setAttr -s 4 \".s\";\n";  


  // Materials hookup
  printf "\n// Connect color shaders with materials\n\n";
  printf "connectAttr \"%slambertdSG.msg\" \"%smaterial.sg\"; \n",
       maya_name,maya_name;
  printf "connectAttr \"%slambert.msg\" \"%smaterial.m\"; \n",
       maya_name,maya_name;
  printf "connectAttr \"%slambert.oc\" \"%slambertSG.ss\"; \n",
       maya_name,maya_name;

  // Connect facets to colors
  printf "\n// Connect mesh to material.\n\n";
  printf "connectAttr \"%sShape.iog\" \"%slambertSG.dsm\" -na;\n",
       maya_name,maya_name;

  // More fiddling with colors
  printf "\n// More connecting up shaders and lights\n\n";
  printf "connectAttr \":defaultLightSet.msg\" \"%slightLinker.lnk[0].llnk\";\n",maya_name;
  printf "connectAttr \":initialShadingGroup.msg\" \"%slightLinker.lnk[0].olnk\";\n",maya_name;
  printf "connectAttr \":defaultLightSet.msg\" \"%slightLinker.slnk[0].sllk\";\n",maya_name;
  printf "connectAttr \":initialShadingGroup.msg\" \"%slightLinker.slnk[0].solk\";\n",maya_name;

  printf "connectAttr \":defaultLightSet.msg\" \"%slightLinker.lnk[1].llnk\";\n",maya_name;
  printf "connectAttr \":initialParticleSE.msg\" \"%slightLinker.lnk[1].olnk\";\n",maya_name;
  printf "connectAttr \":defaultLightSet.msg\" \"%slightLinker.slnk[1].sllk\";\n",maya_name;
  printf "connectAttr \":initialParticleSE.msg\" \"%slightLinker.slnk[1].solk\";\n",maya_name;

  printf "connectAttr \":defaultLightSet.msg\" \"%slightLinker.lnk[%d].llnk\";\n",
       maya_name,2;
  printf "connectAttr \"%slambertSG.msg\" \"%slightLinker.lnk[%d].olnk\";\n",
       maya_name,maya_name,2;
  printf "connectAttr \":defaultLightSet.msg\" \"%slightLinker.slnk[%d].sllk\";\n",
       maya_name,2;
  printf "connectAttr \"%slambertSG.msg\" \"%slightLinker.slnk[%d].solk\";\n",
       maya_name,maya_name,2;
  printf "connectAttr \"%slambertSG.pa\" \":renderPartition.st\" -na;\n",maya_name;
  printf "connectAttr \"%slambert.msg\" \":defaultShaderList1.s\" -na;\n",maya_name;

  printf "\n// And plug lights into global list.\n";
  printf "connectAttr \"%slightLinker.msg\" \":lightList1.ln\" -na;\n",maya_name;


  printf "\n// End of file\n";

} // end maya


/* Usage:
    Set length unit, if not cm.
    Set edge and facet "show" criteria.
    Example:

    Enter command: maya_length_unit := "in"
    Enter command: show facet where not fixed
    Enter command: maya >>> "filename.ma"

   Valid length unit strings are:
    "mm", "millimeter", "cm", "centimeter", "m", "meter", "km", "kilometer", 
    "in", "inch", "ft", "foot", "yd", "yard", "mi", "mile".

*/

