/*
 * Copyright (c) 2019 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef __aarch64__

#include <algorithm>

#include "arm_gemm.hpp"

#include <cstdint>
#include "../../asmlib.hpp"
#include "../../utils.hpp"

namespace arm_gemm {

void a64_smallK_hybrid_u8u32_dot_4x8(const uint8_t *A, int lda, const uint8_t *B, uint32_t *C, int ldc, int M, int N, int K, const uint32_t *, Activation, bool) {
    const long loops_count = iceildiv(N, (int)4) - 1;
    const long ldab = lda * sizeof(uint8_t);
    const long ldcb = ldc * sizeof(uint32_t);
    const long odds_count = K % 4;
    K = (K + 3) / 4;

    for (int y0=0; y0<M; y0+=8) {
        long loops = loops_count;
        long oob_rows = std::max(8 - (M-y0), 0);
        long odds = odds_count;
        const uint8_t *b_ptr0 = B;
        const uint8_t *a_ptr0 = A + (y0 * lda);

        uint32_t *c_ptr0 = C + (y0 * ldc);

        switch(K) {
            case 1:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "cbnz %[odds], 2f\n"
                    "ldr s0, [%[a_ptr0]]\n"
                    "ldr s1, [a_ptr1]\n"
                    "ldr s2, [a_ptr2]\n"
                    "ldr s3, [a_ptr3]\n"
                    "ldr s4, [a_ptr4]\n"
                    "ldr s5, [a_ptr5]\n"
                    "ldr s6, [a_ptr6]\n"
                    "ldr s7, [a_ptr7]\n"
                    "b 3f\n"
                    "2:\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 4f\n"
                    "ldr b0, [%[a_ptr0]]\n"
                    "ldr b1, [a_ptr1]\n"
                    "ldr b2, [a_ptr2]\n"
                    "ldr b3, [a_ptr3]\n"
                    "ldr b4, [a_ptr4]\n"
                    "ldr b5, [a_ptr5]\n"
                    "ldr b6, [a_ptr6]\n"
                    "ldr b7, [a_ptr7]\n"
                    "b 3f\n"
                    "4:\n"
                    "ldr h0, [%[a_ptr0]], #0x2\n"
                    "ldr h1, [a_ptr1], #0x2\n"
                    "ldr h2, [a_ptr2], #0x2\n"
                    "ldr h3, [a_ptr3], #0x2\n"
                    "ldr h4, [a_ptr4], #0x2\n"
                    "ldr h5, [a_ptr5], #0x2\n"
                    "ldr h6, [a_ptr6], #0x2\n"
                    "ldr h7, [a_ptr7], #0x2\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 5f\n"
                    "b 3f\n"
                    "5:\n"
                    "ld1 {v0.b}[2], [%[a_ptr0]]\n"
                    "ld1 {v1.b}[2], [a_ptr1]\n"
                    "ld1 {v2.b}[2], [a_ptr2]\n"
                    "ld1 {v3.b}[2], [a_ptr3]\n"
                    "ld1 {v4.b}[2], [a_ptr4]\n"
                    "ld1 {v5.b}[2], [a_ptr5]\n"
                    "ld1 {v6.b}[2], [a_ptr6]\n"
                    "ld1 {v7.b}[2], [a_ptr7]\n"
                    "3:\n"
                    "movi v24.4s, #0\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "movi v25.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x40]\n"
                    "movi v26.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x80]\n"
                    "movi v27.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0xc0]\n"
                    "movi v28.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x100]\n"
                    "movi v29.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x140]\n"
                    "movi v30.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x180]\n"
                    "movi v31.4s, #0\n"
                    "add %[b_ptr0], %[b_ptr0], #0x10\n"
                    ".inst 0x6f80e218 // udot v24.4s, v16.16b, v0.4b[0]\n"
                    ".inst 0x6f81e219 // udot v25.4s, v16.16b, v1.4b[0]\n"
                    ".inst 0x6f82e21a // udot v26.4s, v16.16b, v2.4b[0]\n"
                    ".inst 0x6f83e21b // udot v27.4s, v16.16b, v3.4b[0]\n"
                    ".inst 0x6f84e21c // udot v28.4s, v16.16b, v4.4b[0]\n"
                    ".inst 0x6f85e21d // udot v29.4s, v16.16b, v5.4b[0]\n"
                    ".inst 0x6f86e21e // udot v30.4s, v16.16b, v6.4b[0]\n"
                    ".inst 0x6f87e21f // udot v31.4s, v16.16b, v7.4b[0]\n"
                    "cbz %[loops], 6f\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "add %[b_ptr0], %[b_ptr0], #0x10\n"
                    "b.eq 7f\n"
                    "8:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "movi v24.4s, #0\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q25, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "movi v25.4s, #0\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    ".inst 0x6f80e218 // udot v24.4s, v16.16b, v0.4b[0]\n"
                    "str q26, [c_ptr2]\n"
                    "movi v26.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x6f81e219 // udot v25.4s, v16.16b, v1.4b[0]\n"
                    "str q27, [c_ptr3]\n"
                    "movi v27.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x6f82e21a // udot v26.4s, v16.16b, v2.4b[0]\n"
                    "str q28, [c_ptr4]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x6f83e21b // udot v27.4s, v16.16b, v3.4b[0]\n"
                    "str q29, [c_ptr5]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x6f84e21c // udot v28.4s, v16.16b, v4.4b[0]\n"
                    "str q30, [c_ptr6]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    ".inst 0x6f85e21d // udot v29.4s, v16.16b, v5.4b[0]\n"
                    "str q31, [c_ptr7]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".inst 0x6f86e21e // udot v30.4s, v16.16b, v6.4b[0]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    ".inst 0x6f87e21f // udot v31.4s, v16.16b, v7.4b[0]\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x10\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    "prfm PSTL1KEEP, [c_ptr6, #0x40]\n"
                    "prfm PSTL1KEEP, [c_ptr7, #0x40]\n"
                    "b.ne 8b\n"
                    "7:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "movi v24.4s, #0\n"
                    "str q25, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "movi v25.4s, #0\n"
                    ".inst 0x6f80e218 // udot v24.4s, v16.16b, v0.4b[0]\n"
                    "str q26, [c_ptr2]\n"
                    "movi v26.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x6f81e219 // udot v25.4s, v16.16b, v1.4b[0]\n"
                    "str q27, [c_ptr3]\n"
                    "movi v27.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x6f82e21a // udot v26.4s, v16.16b, v2.4b[0]\n"
                    "str q28, [c_ptr4]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x6f83e21b // udot v27.4s, v16.16b, v3.4b[0]\n"
                    "str q29, [c_ptr5]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x6f84e21c // udot v28.4s, v16.16b, v4.4b[0]\n"
                    "str q30, [c_ptr6]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    ".inst 0x6f85e21d // udot v29.4s, v16.16b, v5.4b[0]\n"
                    "str q31, [c_ptr7]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".inst 0x6f86e21e // udot v30.4s, v16.16b, v6.4b[0]\n"
                    ".inst 0x6f87e21f // udot v31.4s, v16.16b, v7.4b[0]\n"
                    "6:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q25, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "str q26, [c_ptr2]\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    "str q27, [c_ptr3]\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    "str q28, [c_ptr4]\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    "str q29, [c_ptr5]\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    "str q30, [c_ptr6]\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    "str q31, [c_ptr7]\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "cc", "memory"
                );
                break;
            case 2:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "cbnz %[odds], 2f\n"
                    "ldr d0, [%[a_ptr0]]\n"
                    "ldr d1, [a_ptr1]\n"
                    "ldr d2, [a_ptr2]\n"
                    "ldr d3, [a_ptr3]\n"
                    "ldr d4, [a_ptr4]\n"
                    "ldr d5, [a_ptr5]\n"
                    "ldr d6, [a_ptr6]\n"
                    "ldr d7, [a_ptr7]\n"
                    "b 3f\n"
                    "2:\n"
                    "ldr s0, [%[a_ptr0]], #0x4\n"
                    "ldr s1, [a_ptr1], #0x4\n"
                    "ldr s2, [a_ptr2], #0x4\n"
                    "ldr s3, [a_ptr3], #0x4\n"
                    "ldr s4, [a_ptr4], #0x4\n"
                    "ldr s5, [a_ptr5], #0x4\n"
                    "ldr s6, [a_ptr6], #0x4\n"
                    "ldr s7, [a_ptr7], #0x4\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 4f\n"
                    "ld1 {v0.b}[4], [%[a_ptr0]]\n"
                    "ld1 {v1.b}[4], [a_ptr1]\n"
                    "ld1 {v2.b}[4], [a_ptr2]\n"
                    "ld1 {v3.b}[4], [a_ptr3]\n"
                    "ld1 {v4.b}[4], [a_ptr4]\n"
                    "ld1 {v5.b}[4], [a_ptr5]\n"
                    "ld1 {v6.b}[4], [a_ptr6]\n"
                    "ld1 {v7.b}[4], [a_ptr7]\n"
                    "b 3f\n"
                    "4:\n"
                    "ld1 {v0.h}[2], [%[a_ptr0]], #2\n"
                    "ld1 {v1.h}[2], [a_ptr1], #2\n"
                    "ld1 {v2.h}[2], [a_ptr2], #2\n"
                    "ld1 {v3.h}[2], [a_ptr3], #2\n"
                    "ld1 {v4.h}[2], [a_ptr4], #2\n"
                    "ld1 {v5.h}[2], [a_ptr5], #2\n"
                    "ld1 {v6.h}[2], [a_ptr6], #2\n"
                    "ld1 {v7.h}[2], [a_ptr7], #2\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 5f\n"
                    "b 3f\n"
                    "5:\n"
                    "ld1 {v0.b}[6], [%[a_ptr0]]\n"
                    "ld1 {v1.b}[6], [a_ptr1]\n"
                    "ld1 {v2.b}[6], [a_ptr2]\n"
                    "ld1 {v3.b}[6], [a_ptr3]\n"
                    "ld1 {v4.b}[6], [a_ptr4]\n"
                    "ld1 {v5.b}[6], [a_ptr5]\n"
                    "ld1 {v6.b}[6], [a_ptr6]\n"
                    "ld1 {v7.b}[6], [a_ptr7]\n"
                    "3:\n"
                    "movi v24.4s, #0\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "movi v25.4s, #0\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "movi v26.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x40]\n"
                    "movi v27.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x80]\n"
                    "movi v28.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0xc0]\n"
                    "movi v29.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x100]\n"
                    "movi v30.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x140]\n"
                    "movi v31.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x180]\n"
                    ".inst 0x6f80e218 // udot v24.4s, v16.16b, v0.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    ".inst 0x6f81e219 // udot v25.4s, v16.16b, v1.4b[0]\n"
                    ".inst 0x6f82e21a // udot v26.4s, v16.16b, v2.4b[0]\n"
                    ".inst 0x6f83e21b // udot v27.4s, v16.16b, v3.4b[0]\n"
                    ".inst 0x6f84e21c // udot v28.4s, v16.16b, v4.4b[0]\n"
                    ".inst 0x6f85e21d // udot v29.4s, v16.16b, v5.4b[0]\n"
                    ".inst 0x6f86e21e // udot v30.4s, v16.16b, v6.4b[0]\n"
                    ".inst 0x6f87e21f // udot v31.4s, v16.16b, v7.4b[0]\n"
                    ".inst 0x6fa0e238 // udot v24.4s, v17.16b, v0.4b[1]\n"
                    ".inst 0x6fa1e239 // udot v25.4s, v17.16b, v1.4b[1]\n"
                    ".inst 0x6fa2e23a // udot v26.4s, v17.16b, v2.4b[1]\n"
                    ".inst 0x6fa3e23b // udot v27.4s, v17.16b, v3.4b[1]\n"
                    ".inst 0x6fa4e23c // udot v28.4s, v17.16b, v4.4b[1]\n"
                    ".inst 0x6fa5e23d // udot v29.4s, v17.16b, v5.4b[1]\n"
                    ".inst 0x6fa6e23e // udot v30.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x6fa7e23f // udot v31.4s, v17.16b, v7.4b[1]\n"
                    "cbz %[loops], 6f\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "b.eq 7f\n"
                    "8:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "movi v24.4s, #0\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    "str q25, [c_ptr1]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "movi v25.4s, #0\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    ".inst 0x6f80e218 // udot v24.4s, v16.16b, v0.4b[0]\n"
                    "str q26, [c_ptr2]\n"
                    "movi v26.4s, #0\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    ".inst 0x6f81e219 // udot v25.4s, v16.16b, v1.4b[0]\n"
                    "str q27, [c_ptr3]\n"
                    "movi v27.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x6f82e21a // udot v26.4s, v16.16b, v2.4b[0]\n"
                    "str q28, [c_ptr4]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x6f83e21b // udot v27.4s, v16.16b, v3.4b[0]\n"
                    "str q29, [c_ptr5]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x6f84e21c // udot v28.4s, v16.16b, v4.4b[0]\n"
                    "str q30, [c_ptr6]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x6f85e21d // udot v29.4s, v16.16b, v5.4b[0]\n"
                    "str q31, [c_ptr7]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    ".inst 0x6f86e21e // udot v30.4s, v16.16b, v6.4b[0]\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".inst 0x6f87e21f // udot v31.4s, v16.16b, v7.4b[0]\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    ".inst 0x6fa0e238 // udot v24.4s, v17.16b, v0.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    ".inst 0x6fa1e239 // udot v25.4s, v17.16b, v1.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    ".inst 0x6fa2e23a // udot v26.4s, v17.16b, v2.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    ".inst 0x6fa3e23b // udot v27.4s, v17.16b, v3.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    ".inst 0x6fa4e23c // udot v28.4s, v17.16b, v4.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    ".inst 0x6fa5e23d // udot v29.4s, v17.16b, v5.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr6, #0x40]\n"
                    ".inst 0x6fa6e23e // udot v30.4s, v17.16b, v6.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr7, #0x40]\n"
                    ".inst 0x6fa7e23f // udot v31.4s, v17.16b, v7.4b[1]\n"
                    "b.ne 8b\n"
                    "7:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "movi v24.4s, #0\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x20\n"
                    "str q25, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "movi v25.4s, #0\n"
                    ".inst 0x6f80e218 // udot v24.4s, v16.16b, v0.4b[0]\n"
                    "str q26, [c_ptr2]\n"
                    "movi v26.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x6f81e219 // udot v25.4s, v16.16b, v1.4b[0]\n"
                    "str q27, [c_ptr3]\n"
                    "movi v27.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x6f82e21a // udot v26.4s, v16.16b, v2.4b[0]\n"
                    "str q28, [c_ptr4]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x6f83e21b // udot v27.4s, v16.16b, v3.4b[0]\n"
                    "str q29, [c_ptr5]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x6f84e21c // udot v28.4s, v16.16b, v4.4b[0]\n"
                    "str q30, [c_ptr6]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    ".inst 0x6f85e21d // udot v29.4s, v16.16b, v5.4b[0]\n"
                    "str q31, [c_ptr7]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".inst 0x6f86e21e // udot v30.4s, v16.16b, v6.4b[0]\n"
                    ".inst 0x6fa0e238 // udot v24.4s, v17.16b, v0.4b[1]\n"
                    ".inst 0x6f87e21f // udot v31.4s, v16.16b, v7.4b[0]\n"
                    ".inst 0x6fa1e239 // udot v25.4s, v17.16b, v1.4b[1]\n"
                    ".inst 0x6fa2e23a // udot v26.4s, v17.16b, v2.4b[1]\n"
                    ".inst 0x6fa3e23b // udot v27.4s, v17.16b, v3.4b[1]\n"
                    ".inst 0x6fa4e23c // udot v28.4s, v17.16b, v4.4b[1]\n"
                    ".inst 0x6fa5e23d // udot v29.4s, v17.16b, v5.4b[1]\n"
                    ".inst 0x6fa6e23e // udot v30.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x6fa7e23f // udot v31.4s, v17.16b, v7.4b[1]\n"
                    "6:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q25, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "str q26, [c_ptr2]\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    "str q27, [c_ptr3]\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    "str q28, [c_ptr4]\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    "str q29, [c_ptr5]\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    "str q30, [c_ptr6]\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    "str q31, [c_ptr7]\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "cc", "memory"
                );
                break;
            case 3:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ldr d0, [%[a_ptr0]], #0x8\n"
                    "ldr d1, [a_ptr1], #0x8\n"
                    "ldr d2, [a_ptr2], #0x8\n"
                    "ldr d3, [a_ptr3], #0x8\n"
                    "ldr d4, [a_ptr4], #0x8\n"
                    "ldr d5, [a_ptr5], #0x8\n"
                    "ldr d6, [a_ptr6], #0x8\n"
                    "ldr d7, [a_ptr7], #0x8\n"
                    "cbnz %[odds], 2f\n"
                    "ld1 {v0.s}[2], [%[a_ptr0]]\n"
                    "ld1 {v1.s}[2], [a_ptr1]\n"
                    "ld1 {v2.s}[2], [a_ptr2]\n"
                    "ld1 {v3.s}[2], [a_ptr3]\n"
                    "ld1 {v4.s}[2], [a_ptr4]\n"
                    "ld1 {v5.s}[2], [a_ptr5]\n"
                    "ld1 {v6.s}[2], [a_ptr6]\n"
                    "ld1 {v7.s}[2], [a_ptr7]\n"
                    "b 3f\n"
                    "2:\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 4f\n"
                    "ld1 {v0.b}[8], [%[a_ptr0]]\n"
                    "ld1 {v1.b}[8], [a_ptr1]\n"
                    "ld1 {v2.b}[8], [a_ptr2]\n"
                    "ld1 {v3.b}[8], [a_ptr3]\n"
                    "ld1 {v4.b}[8], [a_ptr4]\n"
                    "ld1 {v5.b}[8], [a_ptr5]\n"
                    "ld1 {v6.b}[8], [a_ptr6]\n"
                    "ld1 {v7.b}[8], [a_ptr7]\n"
                    "b 3f\n"
                    "4:\n"
                    "ld1 {v0.h}[4], [%[a_ptr0]], #2\n"
                    "ld1 {v1.h}[4], [a_ptr1], #2\n"
                    "ld1 {v2.h}[4], [a_ptr2], #2\n"
                    "ld1 {v3.h}[4], [a_ptr3], #2\n"
                    "ld1 {v4.h}[4], [a_ptr4], #2\n"
                    "ld1 {v5.h}[4], [a_ptr5], #2\n"
                    "ld1 {v6.h}[4], [a_ptr6], #2\n"
                    "ld1 {v7.h}[4], [a_ptr7], #2\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 5f\n"
                    "b 3f\n"
                    "5:\n"
                    "ld1 {v0.b}[10], [%[a_ptr0]]\n"
                    "ld1 {v1.b}[10], [a_ptr1]\n"
                    "ld1 {v2.b}[10], [a_ptr2]\n"
                    "ld1 {v3.b}[10], [a_ptr3]\n"
                    "ld1 {v4.b}[10], [a_ptr4]\n"
                    "ld1 {v5.b}[10], [a_ptr5]\n"
                    "ld1 {v6.b}[10], [a_ptr6]\n"
                    "ld1 {v7.b}[10], [a_ptr7]\n"
                    "3:\n"
                    "movi v24.4s, #0\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "movi v25.4s, #0\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "movi v26.4s, #0\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    "movi v27.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x40]\n"
                    "movi v28.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x80]\n"
                    "movi v29.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0xc0]\n"
                    "movi v30.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x100]\n"
                    "movi v31.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x140]\n"
                    ".inst 0x6f80e218 // udot v24.4s, v16.16b, v0.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x180]\n"
                    ".inst 0x6f81e219 // udot v25.4s, v16.16b, v1.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x30\n"
                    ".inst 0x6f82e21a // udot v26.4s, v16.16b, v2.4b[0]\n"
                    ".inst 0x6f83e21b // udot v27.4s, v16.16b, v3.4b[0]\n"
                    ".inst 0x6f84e21c // udot v28.4s, v16.16b, v4.4b[0]\n"
                    ".inst 0x6f85e21d // udot v29.4s, v16.16b, v5.4b[0]\n"
                    ".inst 0x6f86e21e // udot v30.4s, v16.16b, v6.4b[0]\n"
                    ".inst 0x6f87e21f // udot v31.4s, v16.16b, v7.4b[0]\n"
                    ".inst 0x6fa0e238 // udot v24.4s, v17.16b, v0.4b[1]\n"
                    ".inst 0x6fa1e239 // udot v25.4s, v17.16b, v1.4b[1]\n"
                    ".inst 0x6fa2e23a // udot v26.4s, v17.16b, v2.4b[1]\n"
                    ".inst 0x6fa3e23b // udot v27.4s, v17.16b, v3.4b[1]\n"
                    ".inst 0x6fa4e23c // udot v28.4s, v17.16b, v4.4b[1]\n"
                    ".inst 0x6fa5e23d // udot v29.4s, v17.16b, v5.4b[1]\n"
                    ".inst 0x6fa6e23e // udot v30.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x6fa7e23f // udot v31.4s, v17.16b, v7.4b[1]\n"
                    ".inst 0x6f80ea58 // udot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x6f81ea59 // udot v25.4s, v18.16b, v1.4b[2]\n"
                    ".inst 0x6f82ea5a // udot v26.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x6f83ea5b // udot v27.4s, v18.16b, v3.4b[2]\n"
                    ".inst 0x6f84ea5c // udot v28.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x6f85ea5d // udot v29.4s, v18.16b, v5.4b[2]\n"
                    ".inst 0x6f86ea5e // udot v30.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x6f87ea5f // udot v31.4s, v18.16b, v7.4b[2]\n"
                    "cbz %[loops], 6f\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "b.eq 7f\n"
                    "8:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "movi v24.4s, #0\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x30\n"
                    "str q25, [c_ptr1]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "movi v25.4s, #0\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    ".inst 0x6f80e218 // udot v24.4s, v16.16b, v0.4b[0]\n"
                    "str q26, [c_ptr2]\n"
                    "movi v26.4s, #0\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    ".inst 0x6f81e219 // udot v25.4s, v16.16b, v1.4b[0]\n"
                    "str q27, [c_ptr3]\n"
                    "movi v27.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x6f82e21a // udot v26.4s, v16.16b, v2.4b[0]\n"
                    "str q28, [c_ptr4]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x6f83e21b // udot v27.4s, v16.16b, v3.4b[0]\n"
                    "str q29, [c_ptr5]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x6f84e21c // udot v28.4s, v16.16b, v4.4b[0]\n"
                    "str q30, [c_ptr6]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x6f85e21d // udot v29.4s, v16.16b, v5.4b[0]\n"
                    "str q31, [c_ptr7]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    ".inst 0x6f86e21e // udot v30.4s, v16.16b, v6.4b[0]\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".inst 0x6f87e21f // udot v31.4s, v16.16b, v7.4b[0]\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    ".inst 0x6fa0e238 // udot v24.4s, v17.16b, v0.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    ".inst 0x6fa1e239 // udot v25.4s, v17.16b, v1.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    ".inst 0x6fa2e23a // udot v26.4s, v17.16b, v2.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    ".inst 0x6fa3e23b // udot v27.4s, v17.16b, v3.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    ".inst 0x6fa4e23c // udot v28.4s, v17.16b, v4.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    ".inst 0x6fa5e23d // udot v29.4s, v17.16b, v5.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr6, #0x40]\n"
                    ".inst 0x6fa6e23e // udot v30.4s, v17.16b, v6.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr7, #0x40]\n"
                    ".inst 0x6fa7e23f // udot v31.4s, v17.16b, v7.4b[1]\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    ".inst 0x6f80ea58 // udot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x6f81ea59 // udot v25.4s, v18.16b, v1.4b[2]\n"
                    ".inst 0x6f82ea5a // udot v26.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x6f83ea5b // udot v27.4s, v18.16b, v3.4b[2]\n"
                    ".inst 0x6f84ea5c // udot v28.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x6f85ea5d // udot v29.4s, v18.16b, v5.4b[2]\n"
                    ".inst 0x6f86ea5e // udot v30.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x6f87ea5f // udot v31.4s, v18.16b, v7.4b[2]\n"
                    "b.ne 8b\n"
                    "7:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "movi v24.4s, #0\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x30\n"
                    "str q25, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "movi v25.4s, #0\n"
                    ".inst 0x6f80e218 // udot v24.4s, v16.16b, v0.4b[0]\n"
                    "str q26, [c_ptr2]\n"
                    "movi v26.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x6f81e219 // udot v25.4s, v16.16b, v1.4b[0]\n"
                    "str q27, [c_ptr3]\n"
                    "movi v27.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x6f82e21a // udot v26.4s, v16.16b, v2.4b[0]\n"
                    "str q28, [c_ptr4]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x6f83e21b // udot v27.4s, v16.16b, v3.4b[0]\n"
                    "str q29, [c_ptr5]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x6f84e21c // udot v28.4s, v16.16b, v4.4b[0]\n"
                    "str q30, [c_ptr6]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    ".inst 0x6f85e21d // udot v29.4s, v16.16b, v5.4b[0]\n"
                    "str q31, [c_ptr7]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".inst 0x6f86e21e // udot v30.4s, v16.16b, v6.4b[0]\n"
                    ".inst 0x6fa0e238 // udot v24.4s, v17.16b, v0.4b[1]\n"
                    ".inst 0x6f87e21f // udot v31.4s, v16.16b, v7.4b[0]\n"
                    ".inst 0x6fa1e239 // udot v25.4s, v17.16b, v1.4b[1]\n"
                    ".inst 0x6fa2e23a // udot v26.4s, v17.16b, v2.4b[1]\n"
                    ".inst 0x6fa3e23b // udot v27.4s, v17.16b, v3.4b[1]\n"
                    ".inst 0x6fa4e23c // udot v28.4s, v17.16b, v4.4b[1]\n"
                    ".inst 0x6fa5e23d // udot v29.4s, v17.16b, v5.4b[1]\n"
                    ".inst 0x6fa6e23e // udot v30.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x6fa7e23f // udot v31.4s, v17.16b, v7.4b[1]\n"
                    ".inst 0x6f80ea58 // udot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x6f81ea59 // udot v25.4s, v18.16b, v1.4b[2]\n"
                    ".inst 0x6f82ea5a // udot v26.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x6f83ea5b // udot v27.4s, v18.16b, v3.4b[2]\n"
                    ".inst 0x6f84ea5c // udot v28.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x6f85ea5d // udot v29.4s, v18.16b, v5.4b[2]\n"
                    ".inst 0x6f86ea5e // udot v30.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x6f87ea5f // udot v31.4s, v18.16b, v7.4b[2]\n"
                    "6:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q25, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "str q26, [c_ptr2]\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    "str q27, [c_ptr3]\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    "str q28, [c_ptr4]\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    "str q29, [c_ptr5]\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    "str q30, [c_ptr6]\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    "str q31, [c_ptr7]\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "cc", "memory"
                );
                break;
            case 4:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "cbnz %[odds], 2f\n"
                    "ldr q0, [%[a_ptr0]]\n"
                    "ldr q1, [a_ptr1]\n"
                    "ldr q2, [a_ptr2]\n"
                    "ldr q3, [a_ptr3]\n"
                    "ldr q4, [a_ptr4]\n"
                    "ldr q5, [a_ptr5]\n"
                    "ldr q6, [a_ptr6]\n"
                    "ldr q7, [a_ptr7]\n"
                    "b 3f\n"
                    "2:\n"
                    "ldr d0, [%[a_ptr0]], #0x8\n"
                    "ldr d1, [a_ptr1], #0x8\n"
                    "ldr d2, [a_ptr2], #0x8\n"
                    "ldr d3, [a_ptr3], #0x8\n"
                    "ldr d4, [a_ptr4], #0x8\n"
                    "ldr d5, [a_ptr5], #0x8\n"
                    "ldr d6, [a_ptr6], #0x8\n"
                    "ldr d7, [a_ptr7], #0x8\n"
                    "ld1 {v0.s}[2], [%[a_ptr0]], #4\n"
                    "ld1 {v1.s}[2], [a_ptr1], #4\n"
                    "ld1 {v2.s}[2], [a_ptr2], #4\n"
                    "ld1 {v3.s}[2], [a_ptr3], #4\n"
                    "ld1 {v4.s}[2], [a_ptr4], #4\n"
                    "ld1 {v5.s}[2], [a_ptr5], #4\n"
                    "ld1 {v6.s}[2], [a_ptr6], #4\n"
                    "ld1 {v7.s}[2], [a_ptr7], #4\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 4f\n"
                    "ld1 {v0.b}[12], [%[a_ptr0]]\n"
                    "ld1 {v1.b}[12], [a_ptr1]\n"
                    "ld1 {v2.b}[12], [a_ptr2]\n"
                    "ld1 {v3.b}[12], [a_ptr3]\n"
                    "ld1 {v4.b}[12], [a_ptr4]\n"
                    "ld1 {v5.b}[12], [a_ptr5]\n"
                    "ld1 {v6.b}[12], [a_ptr6]\n"
                    "ld1 {v7.b}[12], [a_ptr7]\n"
                    "b 3f\n"
                    "4:\n"
                    "ld1 {v0.h}[6], [%[a_ptr0]], #2\n"
                    "ld1 {v1.h}[6], [a_ptr1], #2\n"
                    "ld1 {v2.h}[6], [a_ptr2], #2\n"
                    "ld1 {v3.h}[6], [a_ptr3], #2\n"
                    "ld1 {v4.h}[6], [a_ptr4], #2\n"
                    "ld1 {v5.h}[6], [a_ptr5], #2\n"
                    "ld1 {v6.h}[6], [a_ptr6], #2\n"
                    "ld1 {v7.h}[6], [a_ptr7], #2\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 5f\n"
                    "b 3f\n"
                    "5:\n"
                    "ld1 {v0.b}[14], [%[a_ptr0]]\n"
                    "ld1 {v1.b}[14], [a_ptr1]\n"
                    "ld1 {v2.b}[14], [a_ptr2]\n"
                    "ld1 {v3.b}[14], [a_ptr3]\n"
                    "ld1 {v4.b}[14], [a_ptr4]\n"
                    "ld1 {v5.b}[14], [a_ptr5]\n"
                    "ld1 {v6.b}[14], [a_ptr6]\n"
                    "ld1 {v7.b}[14], [a_ptr7]\n"
                    "3:\n"
                    "movi v24.4s, #0\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "movi v25.4s, #0\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "movi v26.4s, #0\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    "movi v27.4s, #0\n"
                    "ldr q19, [%[b_ptr0], #0x30]\n"
                    "movi v28.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x40]\n"
                    "movi v29.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x80]\n"
                    "movi v30.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0xc0]\n"
                    "movi v31.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x100]\n"
                    ".inst 0x6f80e218 // udot v24.4s, v16.16b, v0.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x140]\n"
                    ".inst 0x6f81e219 // udot v25.4s, v16.16b, v1.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x180]\n"
                    ".inst 0x6f82e21a // udot v26.4s, v16.16b, v2.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x40\n"
                    ".inst 0x6f83e21b // udot v27.4s, v16.16b, v3.4b[0]\n"
                    ".inst 0x6f84e21c // udot v28.4s, v16.16b, v4.4b[0]\n"
                    ".inst 0x6f85e21d // udot v29.4s, v16.16b, v5.4b[0]\n"
                    ".inst 0x6f86e21e // udot v30.4s, v16.16b, v6.4b[0]\n"
                    ".inst 0x6f87e21f // udot v31.4s, v16.16b, v7.4b[0]\n"
                    ".inst 0x6fa0e238 // udot v24.4s, v17.16b, v0.4b[1]\n"
                    ".inst 0x6fa1e239 // udot v25.4s, v17.16b, v1.4b[1]\n"
                    ".inst 0x6fa2e23a // udot v26.4s, v17.16b, v2.4b[1]\n"
                    ".inst 0x6fa3e23b // udot v27.4s, v17.16b, v3.4b[1]\n"
                    ".inst 0x6fa4e23c // udot v28.4s, v17.16b, v4.4b[1]\n"
                    ".inst 0x6fa5e23d // udot v29.4s, v17.16b, v5.4b[1]\n"
                    ".inst 0x6fa6e23e // udot v30.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x6fa7e23f // udot v31.4s, v17.16b, v7.4b[1]\n"
                    ".inst 0x6f80ea58 // udot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x6f81ea59 // udot v25.4s, v18.16b, v1.4b[2]\n"
                    ".inst 0x6f82ea5a // udot v26.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x6f83ea5b // udot v27.4s, v18.16b, v3.4b[2]\n"
                    ".inst 0x6f84ea5c // udot v28.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x6f85ea5d // udot v29.4s, v18.16b, v5.4b[2]\n"
                    ".inst 0x6f86ea5e // udot v30.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x6f87ea5f // udot v31.4s, v18.16b, v7.4b[2]\n"
                    ".inst 0x6fa0ea78 // udot v24.4s, v19.16b, v0.4b[3]\n"
                    ".inst 0x6fa1ea79 // udot v25.4s, v19.16b, v1.4b[3]\n"
                    ".inst 0x6fa2ea7a // udot v26.4s, v19.16b, v2.4b[3]\n"
                    ".inst 0x6fa3ea7b // udot v27.4s, v19.16b, v3.4b[3]\n"
                    ".inst 0x6fa4ea7c // udot v28.4s, v19.16b, v4.4b[3]\n"
                    ".inst 0x6fa5ea7d // udot v29.4s, v19.16b, v5.4b[3]\n"
                    ".inst 0x6fa6ea7e // udot v30.4s, v19.16b, v6.4b[3]\n"
                    ".inst 0x6fa7ea7f // udot v31.4s, v19.16b, v7.4b[3]\n"
                    "cbz %[loops], 6f\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    "b.eq 7f\n"
                    "8:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "movi v24.4s, #0\n"
                    "ldr q19, [%[b_ptr0], #0x30]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x40\n"
                    "str q25, [c_ptr1]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "movi v25.4s, #0\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    ".inst 0x6f80e218 // udot v24.4s, v16.16b, v0.4b[0]\n"
                    "str q26, [c_ptr2]\n"
                    "movi v26.4s, #0\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    ".inst 0x6f81e219 // udot v25.4s, v16.16b, v1.4b[0]\n"
                    "str q27, [c_ptr3]\n"
                    "movi v27.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x6f82e21a // udot v26.4s, v16.16b, v2.4b[0]\n"
                    "str q28, [c_ptr4]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x6f83e21b // udot v27.4s, v16.16b, v3.4b[0]\n"
                    "str q29, [c_ptr5]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x6f84e21c // udot v28.4s, v16.16b, v4.4b[0]\n"
                    "str q30, [c_ptr6]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x6f85e21d // udot v29.4s, v16.16b, v5.4b[0]\n"
                    "str q31, [c_ptr7]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    ".inst 0x6f86e21e // udot v30.4s, v16.16b, v6.4b[0]\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".inst 0x6f87e21f // udot v31.4s, v16.16b, v7.4b[0]\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    ".inst 0x6fa0e238 // udot v24.4s, v17.16b, v0.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    ".inst 0x6fa1e239 // udot v25.4s, v17.16b, v1.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    ".inst 0x6fa2e23a // udot v26.4s, v17.16b, v2.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    ".inst 0x6fa3e23b // udot v27.4s, v17.16b, v3.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    ".inst 0x6fa4e23c // udot v28.4s, v17.16b, v4.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    ".inst 0x6fa5e23d // udot v29.4s, v17.16b, v5.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr6, #0x40]\n"
                    ".inst 0x6fa6e23e // udot v30.4s, v17.16b, v6.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr7, #0x40]\n"
                    ".inst 0x6fa7e23f // udot v31.4s, v17.16b, v7.4b[1]\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    ".inst 0x6f80ea58 // udot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x6f81ea59 // udot v25.4s, v18.16b, v1.4b[2]\n"
                    ".inst 0x6f82ea5a // udot v26.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x6f83ea5b // udot v27.4s, v18.16b, v3.4b[2]\n"
                    ".inst 0x6f84ea5c // udot v28.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x6f85ea5d // udot v29.4s, v18.16b, v5.4b[2]\n"
                    ".inst 0x6f86ea5e // udot v30.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x6f87ea5f // udot v31.4s, v18.16b, v7.4b[2]\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    ".inst 0x6fa0ea78 // udot v24.4s, v19.16b, v0.4b[3]\n"
                    ".inst 0x6fa1ea79 // udot v25.4s, v19.16b, v1.4b[3]\n"
                    ".inst 0x6fa2ea7a // udot v26.4s, v19.16b, v2.4b[3]\n"
                    ".inst 0x6fa3ea7b // udot v27.4s, v19.16b, v3.4b[3]\n"
                    ".inst 0x6fa4ea7c // udot v28.4s, v19.16b, v4.4b[3]\n"
                    ".inst 0x6fa5ea7d // udot v29.4s, v19.16b, v5.4b[3]\n"
                    ".inst 0x6fa6ea7e // udot v30.4s, v19.16b, v6.4b[3]\n"
                    ".inst 0x6fa7ea7f // udot v31.4s, v19.16b, v7.4b[3]\n"
                    "b.ne 8b\n"
                    "7:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "movi v24.4s, #0\n"
                    "ldr q19, [%[b_ptr0], #0x30]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x40\n"
                    "str q25, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "movi v25.4s, #0\n"
                    ".inst 0x6f80e218 // udot v24.4s, v16.16b, v0.4b[0]\n"
                    "str q26, [c_ptr2]\n"
                    "movi v26.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x6f81e219 // udot v25.4s, v16.16b, v1.4b[0]\n"
                    "str q27, [c_ptr3]\n"
                    "movi v27.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x6f82e21a // udot v26.4s, v16.16b, v2.4b[0]\n"
                    "str q28, [c_ptr4]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x6f83e21b // udot v27.4s, v16.16b, v3.4b[0]\n"
                    "str q29, [c_ptr5]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x6f84e21c // udot v28.4s, v16.16b, v4.4b[0]\n"
                    "str q30, [c_ptr6]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    ".inst 0x6f85e21d // udot v29.4s, v16.16b, v5.4b[0]\n"
                    "str q31, [c_ptr7]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".inst 0x6f86e21e // udot v30.4s, v16.16b, v6.4b[0]\n"
                    ".inst 0x6fa0e238 // udot v24.4s, v17.16b, v0.4b[1]\n"
                    ".inst 0x6f87e21f // udot v31.4s, v16.16b, v7.4b[0]\n"
                    ".inst 0x6fa1e239 // udot v25.4s, v17.16b, v1.4b[1]\n"
                    ".inst 0x6fa2e23a // udot v26.4s, v17.16b, v2.4b[1]\n"
                    ".inst 0x6fa3e23b // udot v27.4s, v17.16b, v3.4b[1]\n"
                    ".inst 0x6fa4e23c // udot v28.4s, v17.16b, v4.4b[1]\n"
                    ".inst 0x6fa5e23d // udot v29.4s, v17.16b, v5.4b[1]\n"
                    ".inst 0x6fa6e23e // udot v30.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x6fa7e23f // udot v31.4s, v17.16b, v7.4b[1]\n"
                    ".inst 0x6f80ea58 // udot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x6f81ea59 // udot v25.4s, v18.16b, v1.4b[2]\n"
                    ".inst 0x6f82ea5a // udot v26.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x6f83ea5b // udot v27.4s, v18.16b, v3.4b[2]\n"
                    ".inst 0x6f84ea5c // udot v28.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x6f85ea5d // udot v29.4s, v18.16b, v5.4b[2]\n"
                    ".inst 0x6f86ea5e // udot v30.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x6f87ea5f // udot v31.4s, v18.16b, v7.4b[2]\n"
                    ".inst 0x6fa0ea78 // udot v24.4s, v19.16b, v0.4b[3]\n"
                    ".inst 0x6fa1ea79 // udot v25.4s, v19.16b, v1.4b[3]\n"
                    ".inst 0x6fa2ea7a // udot v26.4s, v19.16b, v2.4b[3]\n"
                    ".inst 0x6fa3ea7b // udot v27.4s, v19.16b, v3.4b[3]\n"
                    ".inst 0x6fa4ea7c // udot v28.4s, v19.16b, v4.4b[3]\n"
                    ".inst 0x6fa5ea7d // udot v29.4s, v19.16b, v5.4b[3]\n"
                    ".inst 0x6fa6ea7e // udot v30.4s, v19.16b, v6.4b[3]\n"
                    ".inst 0x6fa7ea7f // udot v31.4s, v19.16b, v7.4b[3]\n"
                    "6:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q25, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "str q26, [c_ptr2]\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    "str q27, [c_ptr3]\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    "str q28, [c_ptr4]\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    "str q29, [c_ptr5]\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    "str q30, [c_ptr6]\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    "str q31, [c_ptr7]\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "cc", "memory"
                );
                break;
            case 5:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ldr q0, [%[a_ptr0]], #0x10\n"
                    "ldr q2, [a_ptr1], #0x10\n"
                    "ldr q4, [a_ptr2], #0x10\n"
                    "ldr q6, [a_ptr3], #0x10\n"
                    "ldr q8, [a_ptr4], #0x10\n"
                    "ldr q10, [a_ptr5], #0x10\n"
                    "ldr q12, [a_ptr6], #0x10\n"
                    "ldr q14, [a_ptr7], #0x10\n"
                    "cbnz %[odds], 2f\n"
                    "ldr s1, [%[a_ptr0]]\n"
                    "ldr s3, [a_ptr1]\n"
                    "ldr s5, [a_ptr2]\n"
                    "ldr s7, [a_ptr3]\n"
                    "ldr s9, [a_ptr4]\n"
                    "ldr s11, [a_ptr5]\n"
                    "ldr s13, [a_ptr6]\n"
                    "ldr s15, [a_ptr7]\n"
                    "b 3f\n"
                    "2:\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 4f\n"
                    "ldr b1, [%[a_ptr0]]\n"
                    "ldr b3, [a_ptr1]\n"
                    "ldr b5, [a_ptr2]\n"
                    "ldr b7, [a_ptr3]\n"
                    "ldr b9, [a_ptr4]\n"
                    "ldr b11, [a_ptr5]\n"
                    "ldr b13, [a_ptr6]\n"
                    "ldr b15, [a_ptr7]\n"
                    "b 3f\n"
                    "4:\n"
                    "ldr h1, [%[a_ptr0]], #0x2\n"
                    "ldr h3, [a_ptr1], #0x2\n"
                    "ldr h5, [a_ptr2], #0x2\n"
                    "ldr h7, [a_ptr3], #0x2\n"
                    "ldr h9, [a_ptr4], #0x2\n"
                    "ldr h11, [a_ptr5], #0x2\n"
                    "ldr h13, [a_ptr6], #0x2\n"
                    "ldr h15, [a_ptr7], #0x2\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 5f\n"
                    "b 3f\n"
                    "5:\n"
                    "ld1 {v1.b}[2], [%[a_ptr0]]\n"
                    "ld1 {v3.b}[2], [a_ptr1]\n"
                    "ld1 {v5.b}[2], [a_ptr2]\n"
                    "ld1 {v7.b}[2], [a_ptr3]\n"
                    "ld1 {v9.b}[2], [a_ptr4]\n"
                    "ld1 {v11.b}[2], [a_ptr5]\n"
                    "ld1 {v13.b}[2], [a_ptr6]\n"
                    "ld1 {v15.b}[2], [a_ptr7]\n"
                    "3:\n"
                    "movi v24.4s, #0\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "movi v25.4s, #0\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "movi v26.4s, #0\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    "movi v27.4s, #0\n"
                    "ldr q19, [%[b_ptr0], #0x30]\n"
                    "movi v28.4s, #0\n"
                    "ldr q20, [%[b_ptr0], #0x40]\n"
                    "movi v29.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x40]\n"
                    "movi v30.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x80]\n"
                    "movi v31.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0xc0]\n"
                    ".inst 0x6f80e218 // udot v24.4s, v16.16b, v0.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x100]\n"
                    ".inst 0x6f82e219 // udot v25.4s, v16.16b, v2.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x140]\n"
                    ".inst 0x6f84e21a // udot v26.4s, v16.16b, v4.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x180]\n"
                    ".inst 0x6f86e21b // udot v27.4s, v16.16b, v6.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x50\n"
                    ".inst 0x6f88e21c // udot v28.4s, v16.16b, v8.4b[0]\n"
                    ".inst 0x6f8ae21d // udot v29.4s, v16.16b, v10.4b[0]\n"
                    ".inst 0x6f8ce21e // udot v30.4s, v16.16b, v12.4b[0]\n"
                    ".inst 0x6f8ee21f // udot v31.4s, v16.16b, v14.4b[0]\n"
                    ".inst 0x6fa0e238 // udot v24.4s, v17.16b, v0.4b[1]\n"
                    ".inst 0x6fa2e239 // udot v25.4s, v17.16b, v2.4b[1]\n"
                    ".inst 0x6fa4e23a // udot v26.4s, v17.16b, v4.4b[1]\n"
                    ".inst 0x6fa6e23b // udot v27.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x6fa8e23c // udot v28.4s, v17.16b, v8.4b[1]\n"
                    ".inst 0x6faae23d // udot v29.4s, v17.16b, v10.4b[1]\n"
                    ".inst 0x6face23e // udot v30.4s, v17.16b, v12.4b[1]\n"
                    ".inst 0x6faee23f // udot v31.4s, v17.16b, v14.4b[1]\n"
                    ".inst 0x6f80ea58 // udot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x6f82ea59 // udot v25.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x6f84ea5a // udot v26.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x6f86ea5b // udot v27.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x6f88ea5c // udot v28.4s, v18.16b, v8.4b[2]\n"
                    ".inst 0x6f8aea5d // udot v29.4s, v18.16b, v10.4b[2]\n"
                    ".inst 0x6f8cea5e // udot v30.4s, v18.16b, v12.4b[2]\n"
                    ".inst 0x6f8eea5f // udot v31.4s, v18.16b, v14.4b[2]\n"
                    ".inst 0x6fa0ea78 // udot v24.4s, v19.16b, v0.4b[3]\n"
                    ".inst 0x6fa2ea79 // udot v25.4s, v19.16b, v2.4b[3]\n"
                    ".inst 0x6fa4ea7a // udot v26.4s, v19.16b, v4.4b[3]\n"
                    ".inst 0x6fa6ea7b // udot v27.4s, v19.16b, v6.4b[3]\n"
                    ".inst 0x6fa8ea7c // udot v28.4s, v19.16b, v8.4b[3]\n"
                    ".inst 0x6faaea7d // udot v29.4s, v19.16b, v10.4b[3]\n"
                    ".inst 0x6facea7e // udot v30.4s, v19.16b, v12.4b[3]\n"
                    ".inst 0x6faeea7f // udot v31.4s, v19.16b, v14.4b[3]\n"
                    ".inst 0x6f81e298 // udot v24.4s, v20.16b, v1.4b[0]\n"
                    ".inst 0x6f83e299 // udot v25.4s, v20.16b, v3.4b[0]\n"
                    ".inst 0x6f85e29a // udot v26.4s, v20.16b, v5.4b[0]\n"
                    ".inst 0x6f87e29b // udot v27.4s, v20.16b, v7.4b[0]\n"
                    ".inst 0x6f89e29c // udot v28.4s, v20.16b, v9.4b[0]\n"
                    ".inst 0x6f8be29d // udot v29.4s, v20.16b, v11.4b[0]\n"
                    ".inst 0x6f8de29e // udot v30.4s, v20.16b, v13.4b[0]\n"
                    ".inst 0x6f8fe29f // udot v31.4s, v20.16b, v15.4b[0]\n"
                    "cbz %[loops], 6f\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    "ldr q19, [%[b_ptr0], #0x30]\n"
                    "b.eq 7f\n"
                    "8:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "movi v24.4s, #0\n"
                    "ldr q20, [%[b_ptr0], #0x40]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x50\n"
                    "str q25, [c_ptr1]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "movi v25.4s, #0\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    ".inst 0x6f80e218 // udot v24.4s, v16.16b, v0.4b[0]\n"
                    "str q26, [c_ptr2]\n"
                    "movi v26.4s, #0\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    ".inst 0x6f82e219 // udot v25.4s, v16.16b, v2.4b[0]\n"
                    "str q27, [c_ptr3]\n"
                    "movi v27.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x6f84e21a // udot v26.4s, v16.16b, v4.4b[0]\n"
                    "str q28, [c_ptr4]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x6f86e21b // udot v27.4s, v16.16b, v6.4b[0]\n"
                    "str q29, [c_ptr5]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x6f88e21c // udot v28.4s, v16.16b, v8.4b[0]\n"
                    "str q30, [c_ptr6]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x6f8ae21d // udot v29.4s, v16.16b, v10.4b[0]\n"
                    "str q31, [c_ptr7]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    ".inst 0x6f8ce21e // udot v30.4s, v16.16b, v12.4b[0]\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".inst 0x6f8ee21f // udot v31.4s, v16.16b, v14.4b[0]\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    ".inst 0x6fa0e238 // udot v24.4s, v17.16b, v0.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    ".inst 0x6fa2e239 // udot v25.4s, v17.16b, v2.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    ".inst 0x6fa4e23a // udot v26.4s, v17.16b, v4.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    ".inst 0x6fa6e23b // udot v27.4s, v17.16b, v6.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    ".inst 0x6fa8e23c // udot v28.4s, v17.16b, v8.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    ".inst 0x6faae23d // udot v29.4s, v17.16b, v10.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr6, #0x40]\n"
                    ".inst 0x6face23e // udot v30.4s, v17.16b, v12.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr7, #0x40]\n"
                    ".inst 0x6faee23f // udot v31.4s, v17.16b, v14.4b[1]\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    ".inst 0x6f80ea58 // udot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x6f82ea59 // udot v25.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x6f84ea5a // udot v26.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x6f86ea5b // udot v27.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x6f88ea5c // udot v28.4s, v18.16b, v8.4b[2]\n"
                    ".inst 0x6f8aea5d // udot v29.4s, v18.16b, v10.4b[2]\n"
                    ".inst 0x6f8cea5e // udot v30.4s, v18.16b, v12.4b[2]\n"
                    ".inst 0x6f8eea5f // udot v31.4s, v18.16b, v14.4b[2]\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    ".inst 0x6fa0ea78 // udot v24.4s, v19.16b, v0.4b[3]\n"
                    ".inst 0x6fa2ea79 // udot v25.4s, v19.16b, v2.4b[3]\n"
                    ".inst 0x6fa4ea7a // udot v26.4s, v19.16b, v4.4b[3]\n"
                    ".inst 0x6fa6ea7b // udot v27.4s, v19.16b, v6.4b[3]\n"
                    ".inst 0x6fa8ea7c // udot v28.4s, v19.16b, v8.4b[3]\n"
                    ".inst 0x6faaea7d // udot v29.4s, v19.16b, v10.4b[3]\n"
                    ".inst 0x6facea7e // udot v30.4s, v19.16b, v12.4b[3]\n"
                    ".inst 0x6faeea7f // udot v31.4s, v19.16b, v14.4b[3]\n"
                    "ldr q19, [%[b_ptr0], #0x30]\n"
                    ".inst 0x6f81e298 // udot v24.4s, v20.16b, v1.4b[0]\n"
                    ".inst 0x6f83e299 // udot v25.4s, v20.16b, v3.4b[0]\n"
                    ".inst 0x6f85e29a // udot v26.4s, v20.16b, v5.4b[0]\n"
                    ".inst 0x6f87e29b // udot v27.4s, v20.16b, v7.4b[0]\n"
                    ".inst 0x6f89e29c // udot v28.4s, v20.16b, v9.4b[0]\n"
                    ".inst 0x6f8be29d // udot v29.4s, v20.16b, v11.4b[0]\n"
                    ".inst 0x6f8de29e // udot v30.4s, v20.16b, v13.4b[0]\n"
                    ".inst 0x6f8fe29f // udot v31.4s, v20.16b, v15.4b[0]\n"
                    "b.ne 8b\n"
                    "7:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "movi v24.4s, #0\n"
                    "ldr q20, [%[b_ptr0], #0x40]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x50\n"
                    "str q25, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "movi v25.4s, #0\n"
                    ".inst 0x6f80e218 // udot v24.4s, v16.16b, v0.4b[0]\n"
                    "str q26, [c_ptr2]\n"
                    "movi v26.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x6f82e219 // udot v25.4s, v16.16b, v2.4b[0]\n"
                    "str q27, [c_ptr3]\n"
                    "movi v27.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x6f84e21a // udot v26.4s, v16.16b, v4.4b[0]\n"
                    "str q28, [c_ptr4]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x6f86e21b // udot v27.4s, v16.16b, v6.4b[0]\n"
                    "str q29, [c_ptr5]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x6f88e21c // udot v28.4s, v16.16b, v8.4b[0]\n"
                    "str q30, [c_ptr6]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    ".inst 0x6f8ae21d // udot v29.4s, v16.16b, v10.4b[0]\n"
                    "str q31, [c_ptr7]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".inst 0x6f8ce21e // udot v30.4s, v16.16b, v12.4b[0]\n"
                    ".inst 0x6fa0e238 // udot v24.4s, v17.16b, v0.4b[1]\n"
                    ".inst 0x6f8ee21f // udot v31.4s, v16.16b, v14.4b[0]\n"
                    ".inst 0x6fa2e239 // udot v25.4s, v17.16b, v2.4b[1]\n"
                    ".inst 0x6fa4e23a // udot v26.4s, v17.16b, v4.4b[1]\n"
                    ".inst 0x6fa6e23b // udot v27.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x6fa8e23c // udot v28.4s, v17.16b, v8.4b[1]\n"
                    ".inst 0x6faae23d // udot v29.4s, v17.16b, v10.4b[1]\n"
                    ".inst 0x6face23e // udot v30.4s, v17.16b, v12.4b[1]\n"
                    ".inst 0x6faee23f // udot v31.4s, v17.16b, v14.4b[1]\n"
                    ".inst 0x6f80ea58 // udot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x6f82ea59 // udot v25.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x6f84ea5a // udot v26.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x6f86ea5b // udot v27.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x6f88ea5c // udot v28.4s, v18.16b, v8.4b[2]\n"
                    ".inst 0x6f8aea5d // udot v29.4s, v18.16b, v10.4b[2]\n"
                    ".inst 0x6f8cea5e // udot v30.4s, v18.16b, v12.4b[2]\n"
                    ".inst 0x6f8eea5f // udot v31.4s, v18.16b, v14.4b[2]\n"
                    ".inst 0x6fa0ea78 // udot v24.4s, v19.16b, v0.4b[3]\n"
                    ".inst 0x6fa2ea79 // udot v25.4s, v19.16b, v2.4b[3]\n"
                    ".inst 0x6fa4ea7a // udot v26.4s, v19.16b, v4.4b[3]\n"
                    ".inst 0x6fa6ea7b // udot v27.4s, v19.16b, v6.4b[3]\n"
                    ".inst 0x6fa8ea7c // udot v28.4s, v19.16b, v8.4b[3]\n"
                    ".inst 0x6faaea7d // udot v29.4s, v19.16b, v10.4b[3]\n"
                    ".inst 0x6facea7e // udot v30.4s, v19.16b, v12.4b[3]\n"
                    ".inst 0x6faeea7f // udot v31.4s, v19.16b, v14.4b[3]\n"
                    ".inst 0x6f81e298 // udot v24.4s, v20.16b, v1.4b[0]\n"
                    ".inst 0x6f83e299 // udot v25.4s, v20.16b, v3.4b[0]\n"
                    ".inst 0x6f85e29a // udot v26.4s, v20.16b, v5.4b[0]\n"
                    ".inst 0x6f87e29b // udot v27.4s, v20.16b, v7.4b[0]\n"
                    ".inst 0x6f89e29c // udot v28.4s, v20.16b, v9.4b[0]\n"
                    ".inst 0x6f8be29d // udot v29.4s, v20.16b, v11.4b[0]\n"
                    ".inst 0x6f8de29e // udot v30.4s, v20.16b, v13.4b[0]\n"
                    ".inst 0x6f8fe29f // udot v31.4s, v20.16b, v15.4b[0]\n"
                    "6:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q25, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "str q26, [c_ptr2]\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    "str q27, [c_ptr3]\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    "str q28, [c_ptr4]\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    "str q29, [c_ptr5]\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    "str q30, [c_ptr6]\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    "str q31, [c_ptr7]\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "cc", "memory"
                );
                break;
            case 6:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ldr q0, [%[a_ptr0]], #0x10\n"
                    "ldr q2, [a_ptr1], #0x10\n"
                    "ldr q4, [a_ptr2], #0x10\n"
                    "ldr q6, [a_ptr3], #0x10\n"
                    "ldr q8, [a_ptr4], #0x10\n"
                    "ldr q10, [a_ptr5], #0x10\n"
                    "ldr q12, [a_ptr6], #0x10\n"
                    "ldr q14, [a_ptr7], #0x10\n"
                    "cbnz %[odds], 2f\n"
                    "ldr d1, [%[a_ptr0]]\n"
                    "ldr d3, [a_ptr1]\n"
                    "ldr d5, [a_ptr2]\n"
                    "ldr d7, [a_ptr3]\n"
                    "ldr d9, [a_ptr4]\n"
                    "ldr d11, [a_ptr5]\n"
                    "ldr d13, [a_ptr6]\n"
                    "ldr d15, [a_ptr7]\n"
                    "b 3f\n"
                    "2:\n"
                    "ldr s1, [%[a_ptr0]], #0x4\n"
                    "ldr s3, [a_ptr1], #0x4\n"
                    "ldr s5, [a_ptr2], #0x4\n"
                    "ldr s7, [a_ptr3], #0x4\n"
                    "ldr s9, [a_ptr4], #0x4\n"
                    "ldr s11, [a_ptr5], #0x4\n"
                    "ldr s13, [a_ptr6], #0x4\n"
                    "ldr s15, [a_ptr7], #0x4\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 4f\n"
                    "ld1 {v1.b}[4], [%[a_ptr0]]\n"
                    "ld1 {v3.b}[4], [a_ptr1]\n"
                    "ld1 {v5.b}[4], [a_ptr2]\n"
                    "ld1 {v7.b}[4], [a_ptr3]\n"
                    "ld1 {v9.b}[4], [a_ptr4]\n"
                    "ld1 {v11.b}[4], [a_ptr5]\n"
                    "ld1 {v13.b}[4], [a_ptr6]\n"
                    "ld1 {v15.b}[4], [a_ptr7]\n"
                    "b 3f\n"
                    "4:\n"
                    "ld1 {v1.h}[2], [%[a_ptr0]], #2\n"
                    "ld1 {v3.h}[2], [a_ptr1], #2\n"
                    "ld1 {v5.h}[2], [a_ptr2], #2\n"
                    "ld1 {v7.h}[2], [a_ptr3], #2\n"
                    "ld1 {v9.h}[2], [a_ptr4], #2\n"
                    "ld1 {v11.h}[2], [a_ptr5], #2\n"
                    "ld1 {v13.h}[2], [a_ptr6], #2\n"
                    "ld1 {v15.h}[2], [a_ptr7], #2\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 5f\n"
                    "b 3f\n"
                    "5:\n"
                    "ld1 {v1.b}[6], [%[a_ptr0]]\n"
                    "ld1 {v3.b}[6], [a_ptr1]\n"
                    "ld1 {v5.b}[6], [a_ptr2]\n"
                    "ld1 {v7.b}[6], [a_ptr3]\n"
                    "ld1 {v9.b}[6], [a_ptr4]\n"
                    "ld1 {v11.b}[6], [a_ptr5]\n"
                    "ld1 {v13.b}[6], [a_ptr6]\n"
                    "ld1 {v15.b}[6], [a_ptr7]\n"
                    "3:\n"
                    "movi v24.4s, #0\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "movi v25.4s, #0\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "movi v26.4s, #0\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    "movi v27.4s, #0\n"
                    "ldr q19, [%[b_ptr0], #0x30]\n"
                    "movi v28.4s, #0\n"
                    "ldr q20, [%[b_ptr0], #0x40]\n"
                    "movi v29.4s, #0\n"
                    "ldr q21, [%[b_ptr0], #0x50]\n"
                    "movi v30.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x40]\n"
                    "movi v31.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x80]\n"
                    ".inst 0x6f80e218 // udot v24.4s, v16.16b, v0.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0xc0]\n"
                    ".inst 0x6f82e219 // udot v25.4s, v16.16b, v2.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x100]\n"
                    ".inst 0x6f84e21a // udot v26.4s, v16.16b, v4.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x140]\n"
                    ".inst 0x6f86e21b // udot v27.4s, v16.16b, v6.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x180]\n"
                    ".inst 0x6f88e21c // udot v28.4s, v16.16b, v8.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x60\n"
                    ".inst 0x6f8ae21d // udot v29.4s, v16.16b, v10.4b[0]\n"
                    ".inst 0x6f8ce21e // udot v30.4s, v16.16b, v12.4b[0]\n"
                    ".inst 0x6f8ee21f // udot v31.4s, v16.16b, v14.4b[0]\n"
                    ".inst 0x6fa0e238 // udot v24.4s, v17.16b, v0.4b[1]\n"
                    ".inst 0x6fa2e239 // udot v25.4s, v17.16b, v2.4b[1]\n"
                    ".inst 0x6fa4e23a // udot v26.4s, v17.16b, v4.4b[1]\n"
                    ".inst 0x6fa6e23b // udot v27.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x6fa8e23c // udot v28.4s, v17.16b, v8.4b[1]\n"
                    ".inst 0x6faae23d // udot v29.4s, v17.16b, v10.4b[1]\n"
                    ".inst 0x6face23e // udot v30.4s, v17.16b, v12.4b[1]\n"
                    ".inst 0x6faee23f // udot v31.4s, v17.16b, v14.4b[1]\n"
                    ".inst 0x6f80ea58 // udot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x6f82ea59 // udot v25.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x6f84ea5a // udot v26.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x6f86ea5b // udot v27.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x6f88ea5c // udot v28.4s, v18.16b, v8.4b[2]\n"
                    ".inst 0x6f8aea5d // udot v29.4s, v18.16b, v10.4b[2]\n"
                    ".inst 0x6f8cea5e // udot v30.4s, v18.16b, v12.4b[2]\n"
                    ".inst 0x6f8eea5f // udot v31.4s, v18.16b, v14.4b[2]\n"
                    ".inst 0x6fa0ea78 // udot v24.4s, v19.16b, v0.4b[3]\n"
                    ".inst 0x6fa2ea79 // udot v25.4s, v19.16b, v2.4b[3]\n"
                    ".inst 0x6fa4ea7a // udot v26.4s, v19.16b, v4.4b[3]\n"
                    ".inst 0x6fa6ea7b // udot v27.4s, v19.16b, v6.4b[3]\n"
                    ".inst 0x6fa8ea7c // udot v28.4s, v19.16b, v8.4b[3]\n"
                    ".inst 0x6faaea7d // udot v29.4s, v19.16b, v10.4b[3]\n"
                    ".inst 0x6facea7e // udot v30.4s, v19.16b, v12.4b[3]\n"
                    ".inst 0x6faeea7f // udot v31.4s, v19.16b, v14.4b[3]\n"
                    ".inst 0x6f81e298 // udot v24.4s, v20.16b, v1.4b[0]\n"
                    ".inst 0x6f83e299 // udot v25.4s, v20.16b, v3.4b[0]\n"
                    ".inst 0x6f85e29a // udot v26.4s, v20.16b, v5.4b[0]\n"
                    ".inst 0x6f87e29b // udot v27.4s, v20.16b, v7.4b[0]\n"
                    ".inst 0x6f89e29c // udot v28.4s, v20.16b, v9.4b[0]\n"
                    ".inst 0x6f8be29d // udot v29.4s, v20.16b, v11.4b[0]\n"
                    ".inst 0x6f8de29e // udot v30.4s, v20.16b, v13.4b[0]\n"
                    ".inst 0x6f8fe29f // udot v31.4s, v20.16b, v15.4b[0]\n"
                    ".inst 0x6fa1e2b8 // udot v24.4s, v21.16b, v1.4b[1]\n"
                    ".inst 0x6fa3e2b9 // udot v25.4s, v21.16b, v3.4b[1]\n"
                    ".inst 0x6fa5e2ba // udot v26.4s, v21.16b, v5.4b[1]\n"
                    ".inst 0x6fa7e2bb // udot v27.4s, v21.16b, v7.4b[1]\n"
                    ".inst 0x6fa9e2bc // udot v28.4s, v21.16b, v9.4b[1]\n"
                    ".inst 0x6fabe2bd // udot v29.4s, v21.16b, v11.4b[1]\n"
                    ".inst 0x6fade2be // udot v30.4s, v21.16b, v13.4b[1]\n"
                    ".inst 0x6fafe2bf // udot v31.4s, v21.16b, v15.4b[1]\n"
                    "cbz %[loops], 6f\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    "ldr q19, [%[b_ptr0], #0x30]\n"
                    "ldr q20, [%[b_ptr0], #0x40]\n"
                    "b.eq 7f\n"
                    "8:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "movi v24.4s, #0\n"
                    "ldr q21, [%[b_ptr0], #0x50]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x60\n"
                    "str q25, [c_ptr1]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "movi v25.4s, #0\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    ".inst 0x6f80e218 // udot v24.4s, v16.16b, v0.4b[0]\n"
                    "str q26, [c_ptr2]\n"
                    "movi v26.4s, #0\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    ".inst 0x6f82e219 // udot v25.4s, v16.16b, v2.4b[0]\n"
                    "str q27, [c_ptr3]\n"
                    "movi v27.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x6f84e21a // udot v26.4s, v16.16b, v4.4b[0]\n"
                    "str q28, [c_ptr4]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x6f86e21b // udot v27.4s, v16.16b, v6.4b[0]\n"
                    "str q29, [c_ptr5]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x6f88e21c // udot v28.4s, v16.16b, v8.4b[0]\n"
                    "str q30, [c_ptr6]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x6f8ae21d // udot v29.4s, v16.16b, v10.4b[0]\n"
                    "str q31, [c_ptr7]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    ".inst 0x6f8ce21e // udot v30.4s, v16.16b, v12.4b[0]\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".inst 0x6f8ee21f // udot v31.4s, v16.16b, v14.4b[0]\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    ".inst 0x6fa0e238 // udot v24.4s, v17.16b, v0.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    ".inst 0x6fa2e239 // udot v25.4s, v17.16b, v2.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    ".inst 0x6fa4e23a // udot v26.4s, v17.16b, v4.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    ".inst 0x6fa6e23b // udot v27.4s, v17.16b, v6.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    ".inst 0x6fa8e23c // udot v28.4s, v17.16b, v8.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    ".inst 0x6faae23d // udot v29.4s, v17.16b, v10.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr6, #0x40]\n"
                    ".inst 0x6face23e // udot v30.4s, v17.16b, v12.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr7, #0x40]\n"
                    ".inst 0x6faee23f // udot v31.4s, v17.16b, v14.4b[1]\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    ".inst 0x6f80ea58 // udot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x6f82ea59 // udot v25.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x6f84ea5a // udot v26.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x6f86ea5b // udot v27.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x6f88ea5c // udot v28.4s, v18.16b, v8.4b[2]\n"
                    ".inst 0x6f8aea5d // udot v29.4s, v18.16b, v10.4b[2]\n"
                    ".inst 0x6f8cea5e // udot v30.4s, v18.16b, v12.4b[2]\n"
                    ".inst 0x6f8eea5f // udot v31.4s, v18.16b, v14.4b[2]\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    ".inst 0x6fa0ea78 // udot v24.4s, v19.16b, v0.4b[3]\n"
                    ".inst 0x6fa2ea79 // udot v25.4s, v19.16b, v2.4b[3]\n"
                    ".inst 0x6fa4ea7a // udot v26.4s, v19.16b, v4.4b[3]\n"
                    ".inst 0x6fa6ea7b // udot v27.4s, v19.16b, v6.4b[3]\n"
                    ".inst 0x6fa8ea7c // udot v28.4s, v19.16b, v8.4b[3]\n"
                    ".inst 0x6faaea7d // udot v29.4s, v19.16b, v10.4b[3]\n"
                    ".inst 0x6facea7e // udot v30.4s, v19.16b, v12.4b[3]\n"
                    ".inst 0x6faeea7f // udot v31.4s, v19.16b, v14.4b[3]\n"
                    "ldr q19, [%[b_ptr0], #0x30]\n"
                    ".inst 0x6f81e298 // udot v24.4s, v20.16b, v1.4b[0]\n"
                    ".inst 0x6f83e299 // udot v25.4s, v20.16b, v3.4b[0]\n"
                    ".inst 0x6f85e29a // udot v26.4s, v20.16b, v5.4b[0]\n"
                    ".inst 0x6f87e29b // udot v27.4s, v20.16b, v7.4b[0]\n"
                    ".inst 0x6f89e29c // udot v28.4s, v20.16b, v9.4b[0]\n"
                    ".inst 0x6f8be29d // udot v29.4s, v20.16b, v11.4b[0]\n"
                    ".inst 0x6f8de29e // udot v30.4s, v20.16b, v13.4b[0]\n"
                    ".inst 0x6f8fe29f // udot v31.4s, v20.16b, v15.4b[0]\n"
                    "ldr q20, [%[b_ptr0], #0x40]\n"
                    ".inst 0x6fa1e2b8 // udot v24.4s, v21.16b, v1.4b[1]\n"
                    ".inst 0x6fa3e2b9 // udot v25.4s, v21.16b, v3.4b[1]\n"
                    ".inst 0x6fa5e2ba // udot v26.4s, v21.16b, v5.4b[1]\n"
                    ".inst 0x6fa7e2bb // udot v27.4s, v21.16b, v7.4b[1]\n"
                    ".inst 0x6fa9e2bc // udot v28.4s, v21.16b, v9.4b[1]\n"
                    ".inst 0x6fabe2bd // udot v29.4s, v21.16b, v11.4b[1]\n"
                    ".inst 0x6fade2be // udot v30.4s, v21.16b, v13.4b[1]\n"
                    ".inst 0x6fafe2bf // udot v31.4s, v21.16b, v15.4b[1]\n"
                    "b.ne 8b\n"
                    "7:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "movi v24.4s, #0\n"
                    "ldr q21, [%[b_ptr0], #0x50]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x60\n"
                    "str q25, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "movi v25.4s, #0\n"
                    ".inst 0x6f80e218 // udot v24.4s, v16.16b, v0.4b[0]\n"
                    "str q26, [c_ptr2]\n"
                    "movi v26.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x6f82e219 // udot v25.4s, v16.16b, v2.4b[0]\n"
                    "str q27, [c_ptr3]\n"
                    "movi v27.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x6f84e21a // udot v26.4s, v16.16b, v4.4b[0]\n"
                    "str q28, [c_ptr4]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x6f86e21b // udot v27.4s, v16.16b, v6.4b[0]\n"
                    "str q29, [c_ptr5]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x6f88e21c // udot v28.4s, v16.16b, v8.4b[0]\n"
                    "str q30, [c_ptr6]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    ".inst 0x6f8ae21d // udot v29.4s, v16.16b, v10.4b[0]\n"
                    "str q31, [c_ptr7]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".inst 0x6f8ce21e // udot v30.4s, v16.16b, v12.4b[0]\n"
                    ".inst 0x6fa0e238 // udot v24.4s, v17.16b, v0.4b[1]\n"
                    ".inst 0x6f8ee21f // udot v31.4s, v16.16b, v14.4b[0]\n"
                    ".inst 0x6fa2e239 // udot v25.4s, v17.16b, v2.4b[1]\n"
                    ".inst 0x6fa4e23a // udot v26.4s, v17.16b, v4.4b[1]\n"
                    ".inst 0x6fa6e23b // udot v27.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x6fa8e23c // udot v28.4s, v17.16b, v8.4b[1]\n"
                    ".inst 0x6faae23d // udot v29.4s, v17.16b, v10.4b[1]\n"
                    ".inst 0x6face23e // udot v30.4s, v17.16b, v12.4b[1]\n"
                    ".inst 0x6faee23f // udot v31.4s, v17.16b, v14.4b[1]\n"
                    ".inst 0x6f80ea58 // udot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x6f82ea59 // udot v25.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x6f84ea5a // udot v26.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x6f86ea5b // udot v27.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x6f88ea5c // udot v28.4s, v18.16b, v8.4b[2]\n"
                    ".inst 0x6f8aea5d // udot v29.4s, v18.16b, v10.4b[2]\n"
                    ".inst 0x6f8cea5e // udot v30.4s, v18.16b, v12.4b[2]\n"
                    ".inst 0x6f8eea5f // udot v31.4s, v18.16b, v14.4b[2]\n"
                    ".inst 0x6fa0ea78 // udot v24.4s, v19.16b, v0.4b[3]\n"
                    ".inst 0x6fa2ea79 // udot v25.4s, v19.16b, v2.4b[3]\n"
                    ".inst 0x6fa4ea7a // udot v26.4s, v19.16b, v4.4b[3]\n"
                    ".inst 0x6fa6ea7b // udot v27.4s, v19.16b, v6.4b[3]\n"
                    ".inst 0x6fa8ea7c // udot v28.4s, v19.16b, v8.4b[3]\n"
                    ".inst 0x6faaea7d // udot v29.4s, v19.16b, v10.4b[3]\n"
                    ".inst 0x6facea7e // udot v30.4s, v19.16b, v12.4b[3]\n"
                    ".inst 0x6faeea7f // udot v31.4s, v19.16b, v14.4b[3]\n"
                    ".inst 0x6f81e298 // udot v24.4s, v20.16b, v1.4b[0]\n"
                    ".inst 0x6f83e299 // udot v25.4s, v20.16b, v3.4b[0]\n"
                    ".inst 0x6f85e29a // udot v26.4s, v20.16b, v5.4b[0]\n"
                    ".inst 0x6f87e29b // udot v27.4s, v20.16b, v7.4b[0]\n"
                    ".inst 0x6f89e29c // udot v28.4s, v20.16b, v9.4b[0]\n"
                    ".inst 0x6f8be29d // udot v29.4s, v20.16b, v11.4b[0]\n"
                    ".inst 0x6f8de29e // udot v30.4s, v20.16b, v13.4b[0]\n"
                    ".inst 0x6f8fe29f // udot v31.4s, v20.16b, v15.4b[0]\n"
                    ".inst 0x6fa1e2b8 // udot v24.4s, v21.16b, v1.4b[1]\n"
                    ".inst 0x6fa3e2b9 // udot v25.4s, v21.16b, v3.4b[1]\n"
                    ".inst 0x6fa5e2ba // udot v26.4s, v21.16b, v5.4b[1]\n"
                    ".inst 0x6fa7e2bb // udot v27.4s, v21.16b, v7.4b[1]\n"
                    ".inst 0x6fa9e2bc // udot v28.4s, v21.16b, v9.4b[1]\n"
                    ".inst 0x6fabe2bd // udot v29.4s, v21.16b, v11.4b[1]\n"
                    ".inst 0x6fade2be // udot v30.4s, v21.16b, v13.4b[1]\n"
                    ".inst 0x6fafe2bf // udot v31.4s, v21.16b, v15.4b[1]\n"
                    "6:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q25, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "str q26, [c_ptr2]\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    "str q27, [c_ptr3]\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    "str q28, [c_ptr4]\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    "str q29, [c_ptr5]\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    "str q30, [c_ptr6]\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    "str q31, [c_ptr7]\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "cc", "memory"
                );
                break;
            case 7:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ldr q0, [%[a_ptr0]], #0x10\n"
                    "ldr q2, [a_ptr1], #0x10\n"
                    "ldr q4, [a_ptr2], #0x10\n"
                    "ldr q6, [a_ptr3], #0x10\n"
                    "ldr d1, [%[a_ptr0]], #0x8\n"
                    "ldr q8, [a_ptr4], #0x10\n"
                    "ldr d3, [a_ptr1], #0x8\n"
                    "ldr q10, [a_ptr5], #0x10\n"
                    "ldr d5, [a_ptr2], #0x8\n"
                    "ldr q12, [a_ptr6], #0x10\n"
                    "ldr d7, [a_ptr3], #0x8\n"
                    "ldr q14, [a_ptr7], #0x10\n"
                    "ldr d9, [a_ptr4], #0x8\n"
                    "ldr d11, [a_ptr5], #0x8\n"
                    "ldr d13, [a_ptr6], #0x8\n"
                    "ldr d15, [a_ptr7], #0x8\n"
                    "cbnz %[odds], 2f\n"
                    "ld1 {v1.s}[2], [%[a_ptr0]]\n"
                    "ld1 {v3.s}[2], [a_ptr1]\n"
                    "ld1 {v5.s}[2], [a_ptr2]\n"
                    "ld1 {v7.s}[2], [a_ptr3]\n"
                    "ld1 {v9.s}[2], [a_ptr4]\n"
                    "ld1 {v11.s}[2], [a_ptr5]\n"
                    "ld1 {v13.s}[2], [a_ptr6]\n"
                    "ld1 {v15.s}[2], [a_ptr7]\n"
                    "b 3f\n"
                    "2:\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 4f\n"
                    "ld1 {v1.b}[8], [%[a_ptr0]]\n"
                    "ld1 {v3.b}[8], [a_ptr1]\n"
                    "ld1 {v5.b}[8], [a_ptr2]\n"
                    "ld1 {v7.b}[8], [a_ptr3]\n"
                    "ld1 {v9.b}[8], [a_ptr4]\n"
                    "ld1 {v11.b}[8], [a_ptr5]\n"
                    "ld1 {v13.b}[8], [a_ptr6]\n"
                    "ld1 {v15.b}[8], [a_ptr7]\n"
                    "b 3f\n"
                    "4:\n"
                    "ld1 {v1.h}[4], [%[a_ptr0]], #2\n"
                    "ld1 {v3.h}[4], [a_ptr1], #2\n"
                    "ld1 {v5.h}[4], [a_ptr2], #2\n"
                    "ld1 {v7.h}[4], [a_ptr3], #2\n"
                    "ld1 {v9.h}[4], [a_ptr4], #2\n"
                    "ld1 {v11.h}[4], [a_ptr5], #2\n"
                    "ld1 {v13.h}[4], [a_ptr6], #2\n"
                    "ld1 {v15.h}[4], [a_ptr7], #2\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 5f\n"
                    "b 3f\n"
                    "5:\n"
                    "ld1 {v1.b}[10], [%[a_ptr0]]\n"
                    "ld1 {v3.b}[10], [a_ptr1]\n"
                    "ld1 {v5.b}[10], [a_ptr2]\n"
                    "ld1 {v7.b}[10], [a_ptr3]\n"
                    "ld1 {v9.b}[10], [a_ptr4]\n"
                    "ld1 {v11.b}[10], [a_ptr5]\n"
                    "ld1 {v13.b}[10], [a_ptr6]\n"
                    "ld1 {v15.b}[10], [a_ptr7]\n"
                    "3:\n"
                    "movi v24.4s, #0\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "movi v25.4s, #0\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "movi v26.4s, #0\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    "movi v27.4s, #0\n"
                    "ldr q19, [%[b_ptr0], #0x30]\n"
                    "movi v28.4s, #0\n"
                    "ldr q20, [%[b_ptr0], #0x40]\n"
                    "movi v29.4s, #0\n"
                    "ldr q21, [%[b_ptr0], #0x50]\n"
                    "movi v30.4s, #0\n"
                    "ldr q22, [%[b_ptr0], #0x60]\n"
                    "movi v31.4s, #0\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x40]\n"
                    ".inst 0x6f80e218 // udot v24.4s, v16.16b, v0.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x80]\n"
                    ".inst 0x6f82e219 // udot v25.4s, v16.16b, v2.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0xc0]\n"
                    ".inst 0x6f84e21a // udot v26.4s, v16.16b, v4.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x100]\n"
                    ".inst 0x6f86e21b // udot v27.4s, v16.16b, v6.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x140]\n"
                    ".inst 0x6f88e21c // udot v28.4s, v16.16b, v8.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x180]\n"
                    ".inst 0x6f8ae21d // udot v29.4s, v16.16b, v10.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x70\n"
                    ".inst 0x6f8ce21e // udot v30.4s, v16.16b, v12.4b[0]\n"
                    ".inst 0x6f8ee21f // udot v31.4s, v16.16b, v14.4b[0]\n"
                    ".inst 0x6fa0e238 // udot v24.4s, v17.16b, v0.4b[1]\n"
                    ".inst 0x6fa2e239 // udot v25.4s, v17.16b, v2.4b[1]\n"
                    ".inst 0x6fa4e23a // udot v26.4s, v17.16b, v4.4b[1]\n"
                    ".inst 0x6fa6e23b // udot v27.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x6fa8e23c // udot v28.4s, v17.16b, v8.4b[1]\n"
                    ".inst 0x6faae23d // udot v29.4s, v17.16b, v10.4b[1]\n"
                    ".inst 0x6face23e // udot v30.4s, v17.16b, v12.4b[1]\n"
                    ".inst 0x6faee23f // udot v31.4s, v17.16b, v14.4b[1]\n"
                    ".inst 0x6f80ea58 // udot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x6f82ea59 // udot v25.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x6f84ea5a // udot v26.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x6f86ea5b // udot v27.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x6f88ea5c // udot v28.4s, v18.16b, v8.4b[2]\n"
                    ".inst 0x6f8aea5d // udot v29.4s, v18.16b, v10.4b[2]\n"
                    ".inst 0x6f8cea5e // udot v30.4s, v18.16b, v12.4b[2]\n"
                    ".inst 0x6f8eea5f // udot v31.4s, v18.16b, v14.4b[2]\n"
                    ".inst 0x6fa0ea78 // udot v24.4s, v19.16b, v0.4b[3]\n"
                    ".inst 0x6fa2ea79 // udot v25.4s, v19.16b, v2.4b[3]\n"
                    ".inst 0x6fa4ea7a // udot v26.4s, v19.16b, v4.4b[3]\n"
                    ".inst 0x6fa6ea7b // udot v27.4s, v19.16b, v6.4b[3]\n"
                    ".inst 0x6fa8ea7c // udot v28.4s, v19.16b, v8.4b[3]\n"
                    ".inst 0x6faaea7d // udot v29.4s, v19.16b, v10.4b[3]\n"
                    ".inst 0x6facea7e // udot v30.4s, v19.16b, v12.4b[3]\n"
                    ".inst 0x6faeea7f // udot v31.4s, v19.16b, v14.4b[3]\n"
                    ".inst 0x6f81e298 // udot v24.4s, v20.16b, v1.4b[0]\n"
                    ".inst 0x6f83e299 // udot v25.4s, v20.16b, v3.4b[0]\n"
                    ".inst 0x6f85e29a // udot v26.4s, v20.16b, v5.4b[0]\n"
                    ".inst 0x6f87e29b // udot v27.4s, v20.16b, v7.4b[0]\n"
                    ".inst 0x6f89e29c // udot v28.4s, v20.16b, v9.4b[0]\n"
                    ".inst 0x6f8be29d // udot v29.4s, v20.16b, v11.4b[0]\n"
                    ".inst 0x6f8de29e // udot v30.4s, v20.16b, v13.4b[0]\n"
                    ".inst 0x6f8fe29f // udot v31.4s, v20.16b, v15.4b[0]\n"
                    ".inst 0x6fa1e2b8 // udot v24.4s, v21.16b, v1.4b[1]\n"
                    ".inst 0x6fa3e2b9 // udot v25.4s, v21.16b, v3.4b[1]\n"
                    ".inst 0x6fa5e2ba // udot v26.4s, v21.16b, v5.4b[1]\n"
                    ".inst 0x6fa7e2bb // udot v27.4s, v21.16b, v7.4b[1]\n"
                    ".inst 0x6fa9e2bc // udot v28.4s, v21.16b, v9.4b[1]\n"
                    ".inst 0x6fabe2bd // udot v29.4s, v21.16b, v11.4b[1]\n"
                    ".inst 0x6fade2be // udot v30.4s, v21.16b, v13.4b[1]\n"
                    ".inst 0x6fafe2bf // udot v31.4s, v21.16b, v15.4b[1]\n"
                    ".inst 0x6f81ead8 // udot v24.4s, v22.16b, v1.4b[2]\n"
                    ".inst 0x6f83ead9 // udot v25.4s, v22.16b, v3.4b[2]\n"
                    ".inst 0x6f85eada // udot v26.4s, v22.16b, v5.4b[2]\n"
                    ".inst 0x6f87eadb // udot v27.4s, v22.16b, v7.4b[2]\n"
                    ".inst 0x6f89eadc // udot v28.4s, v22.16b, v9.4b[2]\n"
                    ".inst 0x6f8beadd // udot v29.4s, v22.16b, v11.4b[2]\n"
                    ".inst 0x6f8deade // udot v30.4s, v22.16b, v13.4b[2]\n"
                    ".inst 0x6f8feadf // udot v31.4s, v22.16b, v15.4b[2]\n"
                    "cbz %[loops], 6f\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    "ldr q19, [%[b_ptr0], #0x30]\n"
                    "ldr q20, [%[b_ptr0], #0x40]\n"
                    "ldr q21, [%[b_ptr0], #0x50]\n"
                    "b.eq 7f\n"
                    "8:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "movi v24.4s, #0\n"
                    "ldr q22, [%[b_ptr0], #0x60]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x70\n"
                    "str q25, [c_ptr1]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "movi v25.4s, #0\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    ".inst 0x6f80e218 // udot v24.4s, v16.16b, v0.4b[0]\n"
                    "str q26, [c_ptr2]\n"
                    "movi v26.4s, #0\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    ".inst 0x6f82e219 // udot v25.4s, v16.16b, v2.4b[0]\n"
                    "str q27, [c_ptr3]\n"
                    "movi v27.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x6f84e21a // udot v26.4s, v16.16b, v4.4b[0]\n"
                    "str q28, [c_ptr4]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x6f86e21b // udot v27.4s, v16.16b, v6.4b[0]\n"
                    "str q29, [c_ptr5]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x6f88e21c // udot v28.4s, v16.16b, v8.4b[0]\n"
                    "str q30, [c_ptr6]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x6f8ae21d // udot v29.4s, v16.16b, v10.4b[0]\n"
                    "str q31, [c_ptr7]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    ".inst 0x6f8ce21e // udot v30.4s, v16.16b, v12.4b[0]\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".inst 0x6f8ee21f // udot v31.4s, v16.16b, v14.4b[0]\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    ".inst 0x6fa0e238 // udot v24.4s, v17.16b, v0.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    ".inst 0x6fa2e239 // udot v25.4s, v17.16b, v2.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    ".inst 0x6fa4e23a // udot v26.4s, v17.16b, v4.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    ".inst 0x6fa6e23b // udot v27.4s, v17.16b, v6.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    ".inst 0x6fa8e23c // udot v28.4s, v17.16b, v8.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    ".inst 0x6faae23d // udot v29.4s, v17.16b, v10.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr6, #0x40]\n"
                    ".inst 0x6face23e // udot v30.4s, v17.16b, v12.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr7, #0x40]\n"
                    ".inst 0x6faee23f // udot v31.4s, v17.16b, v14.4b[1]\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    ".inst 0x6f80ea58 // udot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x6f82ea59 // udot v25.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x6f84ea5a // udot v26.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x6f86ea5b // udot v27.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x6f88ea5c // udot v28.4s, v18.16b, v8.4b[2]\n"
                    ".inst 0x6f8aea5d // udot v29.4s, v18.16b, v10.4b[2]\n"
                    ".inst 0x6f8cea5e // udot v30.4s, v18.16b, v12.4b[2]\n"
                    ".inst 0x6f8eea5f // udot v31.4s, v18.16b, v14.4b[2]\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    ".inst 0x6fa0ea78 // udot v24.4s, v19.16b, v0.4b[3]\n"
                    ".inst 0x6fa2ea79 // udot v25.4s, v19.16b, v2.4b[3]\n"
                    ".inst 0x6fa4ea7a // udot v26.4s, v19.16b, v4.4b[3]\n"
                    ".inst 0x6fa6ea7b // udot v27.4s, v19.16b, v6.4b[3]\n"
                    ".inst 0x6fa8ea7c // udot v28.4s, v19.16b, v8.4b[3]\n"
                    ".inst 0x6faaea7d // udot v29.4s, v19.16b, v10.4b[3]\n"
                    ".inst 0x6facea7e // udot v30.4s, v19.16b, v12.4b[3]\n"
                    ".inst 0x6faeea7f // udot v31.4s, v19.16b, v14.4b[3]\n"
                    "ldr q19, [%[b_ptr0], #0x30]\n"
                    ".inst 0x6f81e298 // udot v24.4s, v20.16b, v1.4b[0]\n"
                    ".inst 0x6f83e299 // udot v25.4s, v20.16b, v3.4b[0]\n"
                    ".inst 0x6f85e29a // udot v26.4s, v20.16b, v5.4b[0]\n"
                    ".inst 0x6f87e29b // udot v27.4s, v20.16b, v7.4b[0]\n"
                    ".inst 0x6f89e29c // udot v28.4s, v20.16b, v9.4b[0]\n"
                    ".inst 0x6f8be29d // udot v29.4s, v20.16b, v11.4b[0]\n"
                    ".inst 0x6f8de29e // udot v30.4s, v20.16b, v13.4b[0]\n"
                    ".inst 0x6f8fe29f // udot v31.4s, v20.16b, v15.4b[0]\n"
                    "ldr q20, [%[b_ptr0], #0x40]\n"
                    ".inst 0x6fa1e2b8 // udot v24.4s, v21.16b, v1.4b[1]\n"
                    ".inst 0x6fa3e2b9 // udot v25.4s, v21.16b, v3.4b[1]\n"
                    ".inst 0x6fa5e2ba // udot v26.4s, v21.16b, v5.4b[1]\n"
                    ".inst 0x6fa7e2bb // udot v27.4s, v21.16b, v7.4b[1]\n"
                    ".inst 0x6fa9e2bc // udot v28.4s, v21.16b, v9.4b[1]\n"
                    ".inst 0x6fabe2bd // udot v29.4s, v21.16b, v11.4b[1]\n"
                    ".inst 0x6fade2be // udot v30.4s, v21.16b, v13.4b[1]\n"
                    ".inst 0x6fafe2bf // udot v31.4s, v21.16b, v15.4b[1]\n"
                    "ldr q21, [%[b_ptr0], #0x50]\n"
                    ".inst 0x6f81ead8 // udot v24.4s, v22.16b, v1.4b[2]\n"
                    ".inst 0x6f83ead9 // udot v25.4s, v22.16b, v3.4b[2]\n"
                    ".inst 0x6f85eada // udot v26.4s, v22.16b, v5.4b[2]\n"
                    ".inst 0x6f87eadb // udot v27.4s, v22.16b, v7.4b[2]\n"
                    ".inst 0x6f89eadc // udot v28.4s, v22.16b, v9.4b[2]\n"
                    ".inst 0x6f8beadd // udot v29.4s, v22.16b, v11.4b[2]\n"
                    ".inst 0x6f8deade // udot v30.4s, v22.16b, v13.4b[2]\n"
                    ".inst 0x6f8feadf // udot v31.4s, v22.16b, v15.4b[2]\n"
                    "b.ne 8b\n"
                    "7:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "movi v24.4s, #0\n"
                    "ldr q22, [%[b_ptr0], #0x60]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x70\n"
                    "str q25, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "movi v25.4s, #0\n"
                    ".inst 0x6f80e218 // udot v24.4s, v16.16b, v0.4b[0]\n"
                    "str q26, [c_ptr2]\n"
                    "movi v26.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x6f82e219 // udot v25.4s, v16.16b, v2.4b[0]\n"
                    "str q27, [c_ptr3]\n"
                    "movi v27.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x6f84e21a // udot v26.4s, v16.16b, v4.4b[0]\n"
                    "str q28, [c_ptr4]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x6f86e21b // udot v27.4s, v16.16b, v6.4b[0]\n"
                    "str q29, [c_ptr5]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x6f88e21c // udot v28.4s, v16.16b, v8.4b[0]\n"
                    "str q30, [c_ptr6]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    ".inst 0x6f8ae21d // udot v29.4s, v16.16b, v10.4b[0]\n"
                    "str q31, [c_ptr7]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".inst 0x6f8ce21e // udot v30.4s, v16.16b, v12.4b[0]\n"
                    ".inst 0x6fa0e238 // udot v24.4s, v17.16b, v0.4b[1]\n"
                    ".inst 0x6f8ee21f // udot v31.4s, v16.16b, v14.4b[0]\n"
                    ".inst 0x6fa2e239 // udot v25.4s, v17.16b, v2.4b[1]\n"
                    ".inst 0x6fa4e23a // udot v26.4s, v17.16b, v4.4b[1]\n"
                    ".inst 0x6fa6e23b // udot v27.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x6fa8e23c // udot v28.4s, v17.16b, v8.4b[1]\n"
                    ".inst 0x6faae23d // udot v29.4s, v17.16b, v10.4b[1]\n"
                    ".inst 0x6face23e // udot v30.4s, v17.16b, v12.4b[1]\n"
                    ".inst 0x6faee23f // udot v31.4s, v17.16b, v14.4b[1]\n"
                    ".inst 0x6f80ea58 // udot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x6f82ea59 // udot v25.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x6f84ea5a // udot v26.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x6f86ea5b // udot v27.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x6f88ea5c // udot v28.4s, v18.16b, v8.4b[2]\n"
                    ".inst 0x6f8aea5d // udot v29.4s, v18.16b, v10.4b[2]\n"
                    ".inst 0x6f8cea5e // udot v30.4s, v18.16b, v12.4b[2]\n"
                    ".inst 0x6f8eea5f // udot v31.4s, v18.16b, v14.4b[2]\n"
                    ".inst 0x6fa0ea78 // udot v24.4s, v19.16b, v0.4b[3]\n"
                    ".inst 0x6fa2ea79 // udot v25.4s, v19.16b, v2.4b[3]\n"
                    ".inst 0x6fa4ea7a // udot v26.4s, v19.16b, v4.4b[3]\n"
                    ".inst 0x6fa6ea7b // udot v27.4s, v19.16b, v6.4b[3]\n"
                    ".inst 0x6fa8ea7c // udot v28.4s, v19.16b, v8.4b[3]\n"
                    ".inst 0x6faaea7d // udot v29.4s, v19.16b, v10.4b[3]\n"
                    ".inst 0x6facea7e // udot v30.4s, v19.16b, v12.4b[3]\n"
                    ".inst 0x6faeea7f // udot v31.4s, v19.16b, v14.4b[3]\n"
                    ".inst 0x6f81e298 // udot v24.4s, v20.16b, v1.4b[0]\n"
                    ".inst 0x6f83e299 // udot v25.4s, v20.16b, v3.4b[0]\n"
                    ".inst 0x6f85e29a // udot v26.4s, v20.16b, v5.4b[0]\n"
                    ".inst 0x6f87e29b // udot v27.4s, v20.16b, v7.4b[0]\n"
                    ".inst 0x6f89e29c // udot v28.4s, v20.16b, v9.4b[0]\n"
                    ".inst 0x6f8be29d // udot v29.4s, v20.16b, v11.4b[0]\n"
                    ".inst 0x6f8de29e // udot v30.4s, v20.16b, v13.4b[0]\n"
                    ".inst 0x6f8fe29f // udot v31.4s, v20.16b, v15.4b[0]\n"
                    ".inst 0x6fa1e2b8 // udot v24.4s, v21.16b, v1.4b[1]\n"
                    ".inst 0x6fa3e2b9 // udot v25.4s, v21.16b, v3.4b[1]\n"
                    ".inst 0x6fa5e2ba // udot v26.4s, v21.16b, v5.4b[1]\n"
                    ".inst 0x6fa7e2bb // udot v27.4s, v21.16b, v7.4b[1]\n"
                    ".inst 0x6fa9e2bc // udot v28.4s, v21.16b, v9.4b[1]\n"
                    ".inst 0x6fabe2bd // udot v29.4s, v21.16b, v11.4b[1]\n"
                    ".inst 0x6fade2be // udot v30.4s, v21.16b, v13.4b[1]\n"
                    ".inst 0x6fafe2bf // udot v31.4s, v21.16b, v15.4b[1]\n"
                    ".inst 0x6f81ead8 // udot v24.4s, v22.16b, v1.4b[2]\n"
                    ".inst 0x6f83ead9 // udot v25.4s, v22.16b, v3.4b[2]\n"
                    ".inst 0x6f85eada // udot v26.4s, v22.16b, v5.4b[2]\n"
                    ".inst 0x6f87eadb // udot v27.4s, v22.16b, v7.4b[2]\n"
                    ".inst 0x6f89eadc // udot v28.4s, v22.16b, v9.4b[2]\n"
                    ".inst 0x6f8beadd // udot v29.4s, v22.16b, v11.4b[2]\n"
                    ".inst 0x6f8deade // udot v30.4s, v22.16b, v13.4b[2]\n"
                    ".inst 0x6f8feadf // udot v31.4s, v22.16b, v15.4b[2]\n"
                    "6:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q25, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "str q26, [c_ptr2]\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    "str q27, [c_ptr3]\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    "str q28, [c_ptr4]\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    "str q29, [c_ptr5]\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    "str q30, [c_ptr6]\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    "str q31, [c_ptr7]\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "cc", "memory"
                );
                break;
            default:
            case 8:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "ldr q0, [%[a_ptr0]], #0x10\n"
                    "ldr q2, [a_ptr1], #0x10\n"
                    "ldr q4, [a_ptr2], #0x10\n"
                    "ldr q6, [a_ptr3], #0x10\n"
                    "ldr q8, [a_ptr4], #0x10\n"
                    "ldr q10, [a_ptr5], #0x10\n"
                    "ldr q12, [a_ptr6], #0x10\n"
                    "ldr q14, [a_ptr7], #0x10\n"
                    "cbnz %[odds], 2f\n"
                    "ldr q1, [%[a_ptr0]]\n"
                    "ldr q3, [a_ptr1]\n"
                    "ldr q5, [a_ptr2]\n"
                    "ldr q7, [a_ptr3]\n"
                    "ldr q9, [a_ptr4]\n"
                    "ldr q11, [a_ptr5]\n"
                    "ldr q13, [a_ptr6]\n"
                    "ldr q15, [a_ptr7]\n"
                    "b 3f\n"
                    "2:\n"
                    "ldr d1, [%[a_ptr0]], #0x8\n"
                    "ldr d3, [a_ptr1], #0x8\n"
                    "ldr d5, [a_ptr2], #0x8\n"
                    "ldr d7, [a_ptr3], #0x8\n"
                    "ldr d9, [a_ptr4], #0x8\n"
                    "ldr d11, [a_ptr5], #0x8\n"
                    "ldr d13, [a_ptr6], #0x8\n"
                    "ldr d15, [a_ptr7], #0x8\n"
                    "ld1 {v1.s}[2], [%[a_ptr0]], #4\n"
                    "ld1 {v3.s}[2], [a_ptr1], #4\n"
                    "ld1 {v5.s}[2], [a_ptr2], #4\n"
                    "ld1 {v7.s}[2], [a_ptr3], #4\n"
                    "ld1 {v9.s}[2], [a_ptr4], #4\n"
                    "ld1 {v11.s}[2], [a_ptr5], #4\n"
                    "ld1 {v13.s}[2], [a_ptr6], #4\n"
                    "ld1 {v15.s}[2], [a_ptr7], #4\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 4f\n"
                    "ld1 {v1.b}[12], [%[a_ptr0]]\n"
                    "ld1 {v3.b}[12], [a_ptr1]\n"
                    "ld1 {v5.b}[12], [a_ptr2]\n"
                    "ld1 {v7.b}[12], [a_ptr3]\n"
                    "ld1 {v9.b}[12], [a_ptr4]\n"
                    "ld1 {v11.b}[12], [a_ptr5]\n"
                    "ld1 {v13.b}[12], [a_ptr6]\n"
                    "ld1 {v15.b}[12], [a_ptr7]\n"
                    "b 3f\n"
                    "4:\n"
                    "ld1 {v1.h}[6], [%[a_ptr0]], #2\n"
                    "ld1 {v3.h}[6], [a_ptr1], #2\n"
                    "ld1 {v5.h}[6], [a_ptr2], #2\n"
                    "ld1 {v7.h}[6], [a_ptr3], #2\n"
                    "ld1 {v9.h}[6], [a_ptr4], #2\n"
                    "ld1 {v11.h}[6], [a_ptr5], #2\n"
                    "ld1 {v13.h}[6], [a_ptr6], #2\n"
                    "ld1 {v15.h}[6], [a_ptr7], #2\n"
                    "subs %[odds], %[odds], #0x1\n"
                    "b.ne 5f\n"
                    "b 3f\n"
                    "5:\n"
                    "ld1 {v1.b}[14], [%[a_ptr0]]\n"
                    "ld1 {v3.b}[14], [a_ptr1]\n"
                    "ld1 {v5.b}[14], [a_ptr2]\n"
                    "ld1 {v7.b}[14], [a_ptr3]\n"
                    "ld1 {v9.b}[14], [a_ptr4]\n"
                    "ld1 {v11.b}[14], [a_ptr5]\n"
                    "ld1 {v13.b}[14], [a_ptr6]\n"
                    "ld1 {v15.b}[14], [a_ptr7]\n"
                    "3:\n"
                    "movi v24.4s, #0\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "movi v25.4s, #0\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "movi v26.4s, #0\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    "movi v27.4s, #0\n"
                    "ldr q19, [%[b_ptr0], #0x30]\n"
                    "movi v28.4s, #0\n"
                    "ldr q20, [%[b_ptr0], #0x40]\n"
                    "movi v29.4s, #0\n"
                    "ldr q21, [%[b_ptr0], #0x50]\n"
                    "movi v30.4s, #0\n"
                    "ldr q22, [%[b_ptr0], #0x60]\n"
                    "movi v31.4s, #0\n"
                    "ldr q23, [%[b_ptr0], #0x70]\n"
                    ".inst 0x6f80e218 // udot v24.4s, v16.16b, v0.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x40]\n"
                    ".inst 0x6f82e219 // udot v25.4s, v16.16b, v2.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x80]\n"
                    ".inst 0x6f84e21a // udot v26.4s, v16.16b, v4.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0xc0]\n"
                    ".inst 0x6f86e21b // udot v27.4s, v16.16b, v6.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x100]\n"
                    ".inst 0x6f88e21c // udot v28.4s, v16.16b, v8.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x140]\n"
                    ".inst 0x6f8ae21d // udot v29.4s, v16.16b, v10.4b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x180]\n"
                    ".inst 0x6f8ce21e // udot v30.4s, v16.16b, v12.4b[0]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x80\n"
                    ".inst 0x6f8ee21f // udot v31.4s, v16.16b, v14.4b[0]\n"
                    ".inst 0x6fa0e238 // udot v24.4s, v17.16b, v0.4b[1]\n"
                    ".inst 0x6fa2e239 // udot v25.4s, v17.16b, v2.4b[1]\n"
                    ".inst 0x6fa4e23a // udot v26.4s, v17.16b, v4.4b[1]\n"
                    ".inst 0x6fa6e23b // udot v27.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x6fa8e23c // udot v28.4s, v17.16b, v8.4b[1]\n"
                    ".inst 0x6faae23d // udot v29.4s, v17.16b, v10.4b[1]\n"
                    ".inst 0x6face23e // udot v30.4s, v17.16b, v12.4b[1]\n"
                    ".inst 0x6faee23f // udot v31.4s, v17.16b, v14.4b[1]\n"
                    ".inst 0x6f80ea58 // udot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x6f82ea59 // udot v25.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x6f84ea5a // udot v26.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x6f86ea5b // udot v27.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x6f88ea5c // udot v28.4s, v18.16b, v8.4b[2]\n"
                    ".inst 0x6f8aea5d // udot v29.4s, v18.16b, v10.4b[2]\n"
                    ".inst 0x6f8cea5e // udot v30.4s, v18.16b, v12.4b[2]\n"
                    ".inst 0x6f8eea5f // udot v31.4s, v18.16b, v14.4b[2]\n"
                    ".inst 0x6fa0ea78 // udot v24.4s, v19.16b, v0.4b[3]\n"
                    ".inst 0x6fa2ea79 // udot v25.4s, v19.16b, v2.4b[3]\n"
                    ".inst 0x6fa4ea7a // udot v26.4s, v19.16b, v4.4b[3]\n"
                    ".inst 0x6fa6ea7b // udot v27.4s, v19.16b, v6.4b[3]\n"
                    ".inst 0x6fa8ea7c // udot v28.4s, v19.16b, v8.4b[3]\n"
                    ".inst 0x6faaea7d // udot v29.4s, v19.16b, v10.4b[3]\n"
                    ".inst 0x6facea7e // udot v30.4s, v19.16b, v12.4b[3]\n"
                    ".inst 0x6faeea7f // udot v31.4s, v19.16b, v14.4b[3]\n"
                    ".inst 0x6f81e298 // udot v24.4s, v20.16b, v1.4b[0]\n"
                    ".inst 0x6f83e299 // udot v25.4s, v20.16b, v3.4b[0]\n"
                    ".inst 0x6f85e29a // udot v26.4s, v20.16b, v5.4b[0]\n"
                    ".inst 0x6f87e29b // udot v27.4s, v20.16b, v7.4b[0]\n"
                    ".inst 0x6f89e29c // udot v28.4s, v20.16b, v9.4b[0]\n"
                    ".inst 0x6f8be29d // udot v29.4s, v20.16b, v11.4b[0]\n"
                    ".inst 0x6f8de29e // udot v30.4s, v20.16b, v13.4b[0]\n"
                    ".inst 0x6f8fe29f // udot v31.4s, v20.16b, v15.4b[0]\n"
                    ".inst 0x6fa1e2b8 // udot v24.4s, v21.16b, v1.4b[1]\n"
                    ".inst 0x6fa3e2b9 // udot v25.4s, v21.16b, v3.4b[1]\n"
                    ".inst 0x6fa5e2ba // udot v26.4s, v21.16b, v5.4b[1]\n"
                    ".inst 0x6fa7e2bb // udot v27.4s, v21.16b, v7.4b[1]\n"
                    ".inst 0x6fa9e2bc // udot v28.4s, v21.16b, v9.4b[1]\n"
                    ".inst 0x6fabe2bd // udot v29.4s, v21.16b, v11.4b[1]\n"
                    ".inst 0x6fade2be // udot v30.4s, v21.16b, v13.4b[1]\n"
                    ".inst 0x6fafe2bf // udot v31.4s, v21.16b, v15.4b[1]\n"
                    ".inst 0x6f81ead8 // udot v24.4s, v22.16b, v1.4b[2]\n"
                    ".inst 0x6f83ead9 // udot v25.4s, v22.16b, v3.4b[2]\n"
                    ".inst 0x6f85eada // udot v26.4s, v22.16b, v5.4b[2]\n"
                    ".inst 0x6f87eadb // udot v27.4s, v22.16b, v7.4b[2]\n"
                    ".inst 0x6f89eadc // udot v28.4s, v22.16b, v9.4b[2]\n"
                    ".inst 0x6f8beadd // udot v29.4s, v22.16b, v11.4b[2]\n"
                    ".inst 0x6f8deade // udot v30.4s, v22.16b, v13.4b[2]\n"
                    ".inst 0x6f8feadf // udot v31.4s, v22.16b, v15.4b[2]\n"
                    ".inst 0x6fa1eaf8 // udot v24.4s, v23.16b, v1.4b[3]\n"
                    ".inst 0x6fa3eaf9 // udot v25.4s, v23.16b, v3.4b[3]\n"
                    ".inst 0x6fa5eafa // udot v26.4s, v23.16b, v5.4b[3]\n"
                    ".inst 0x6fa7eafb // udot v27.4s, v23.16b, v7.4b[3]\n"
                    ".inst 0x6fa9eafc // udot v28.4s, v23.16b, v9.4b[3]\n"
                    ".inst 0x6fabeafd // udot v29.4s, v23.16b, v11.4b[3]\n"
                    ".inst 0x6fadeafe // udot v30.4s, v23.16b, v13.4b[3]\n"
                    ".inst 0x6fafeaff // udot v31.4s, v23.16b, v15.4b[3]\n"
                    "cbz %[loops], 6f\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    "ldr q19, [%[b_ptr0], #0x30]\n"
                    "ldr q20, [%[b_ptr0], #0x40]\n"
                    "ldr q21, [%[b_ptr0], #0x50]\n"
                    "ldr q22, [%[b_ptr0], #0x60]\n"
                    "b.eq 7f\n"
                    "8:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "movi v24.4s, #0\n"
                    "ldr q23, [%[b_ptr0], #0x70]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x80\n"
                    "str q25, [c_ptr1]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "movi v25.4s, #0\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    ".inst 0x6f80e218 // udot v24.4s, v16.16b, v0.4b[0]\n"
                    "str q26, [c_ptr2]\n"
                    "movi v26.4s, #0\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    ".inst 0x6f82e219 // udot v25.4s, v16.16b, v2.4b[0]\n"
                    "str q27, [c_ptr3]\n"
                    "movi v27.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x6f84e21a // udot v26.4s, v16.16b, v4.4b[0]\n"
                    "str q28, [c_ptr4]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x6f86e21b // udot v27.4s, v16.16b, v6.4b[0]\n"
                    "str q29, [c_ptr5]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x6f88e21c // udot v28.4s, v16.16b, v8.4b[0]\n"
                    "str q30, [c_ptr6]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x6f8ae21d // udot v29.4s, v16.16b, v10.4b[0]\n"
                    "str q31, [c_ptr7]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    ".inst 0x6f8ce21e // udot v30.4s, v16.16b, v12.4b[0]\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".inst 0x6f8ee21f // udot v31.4s, v16.16b, v14.4b[0]\n"
                    "ldr q16, [%[b_ptr0]]\n"
                    ".inst 0x6fa0e238 // udot v24.4s, v17.16b, v0.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    ".inst 0x6fa2e239 // udot v25.4s, v17.16b, v2.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    ".inst 0x6fa4e23a // udot v26.4s, v17.16b, v4.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    ".inst 0x6fa6e23b // udot v27.4s, v17.16b, v6.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    ".inst 0x6fa8e23c // udot v28.4s, v17.16b, v8.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    ".inst 0x6faae23d // udot v29.4s, v17.16b, v10.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr6, #0x40]\n"
                    ".inst 0x6face23e // udot v30.4s, v17.16b, v12.4b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr7, #0x40]\n"
                    ".inst 0x6faee23f // udot v31.4s, v17.16b, v14.4b[1]\n"
                    "ldr q17, [%[b_ptr0], #0x10]\n"
                    ".inst 0x6f80ea58 // udot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x6f82ea59 // udot v25.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x6f84ea5a // udot v26.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x6f86ea5b // udot v27.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x6f88ea5c // udot v28.4s, v18.16b, v8.4b[2]\n"
                    ".inst 0x6f8aea5d // udot v29.4s, v18.16b, v10.4b[2]\n"
                    ".inst 0x6f8cea5e // udot v30.4s, v18.16b, v12.4b[2]\n"
                    ".inst 0x6f8eea5f // udot v31.4s, v18.16b, v14.4b[2]\n"
                    "ldr q18, [%[b_ptr0], #0x20]\n"
                    ".inst 0x6fa0ea78 // udot v24.4s, v19.16b, v0.4b[3]\n"
                    ".inst 0x6fa2ea79 // udot v25.4s, v19.16b, v2.4b[3]\n"
                    ".inst 0x6fa4ea7a // udot v26.4s, v19.16b, v4.4b[3]\n"
                    ".inst 0x6fa6ea7b // udot v27.4s, v19.16b, v6.4b[3]\n"
                    ".inst 0x6fa8ea7c // udot v28.4s, v19.16b, v8.4b[3]\n"
                    ".inst 0x6faaea7d // udot v29.4s, v19.16b, v10.4b[3]\n"
                    ".inst 0x6facea7e // udot v30.4s, v19.16b, v12.4b[3]\n"
                    ".inst 0x6faeea7f // udot v31.4s, v19.16b, v14.4b[3]\n"
                    "ldr q19, [%[b_ptr0], #0x30]\n"
                    ".inst 0x6f81e298 // udot v24.4s, v20.16b, v1.4b[0]\n"
                    ".inst 0x6f83e299 // udot v25.4s, v20.16b, v3.4b[0]\n"
                    ".inst 0x6f85e29a // udot v26.4s, v20.16b, v5.4b[0]\n"
                    ".inst 0x6f87e29b // udot v27.4s, v20.16b, v7.4b[0]\n"
                    ".inst 0x6f89e29c // udot v28.4s, v20.16b, v9.4b[0]\n"
                    ".inst 0x6f8be29d // udot v29.4s, v20.16b, v11.4b[0]\n"
                    ".inst 0x6f8de29e // udot v30.4s, v20.16b, v13.4b[0]\n"
                    ".inst 0x6f8fe29f // udot v31.4s, v20.16b, v15.4b[0]\n"
                    "ldr q20, [%[b_ptr0], #0x40]\n"
                    ".inst 0x6fa1e2b8 // udot v24.4s, v21.16b, v1.4b[1]\n"
                    ".inst 0x6fa3e2b9 // udot v25.4s, v21.16b, v3.4b[1]\n"
                    ".inst 0x6fa5e2ba // udot v26.4s, v21.16b, v5.4b[1]\n"
                    ".inst 0x6fa7e2bb // udot v27.4s, v21.16b, v7.4b[1]\n"
                    ".inst 0x6fa9e2bc // udot v28.4s, v21.16b, v9.4b[1]\n"
                    ".inst 0x6fabe2bd // udot v29.4s, v21.16b, v11.4b[1]\n"
                    ".inst 0x6fade2be // udot v30.4s, v21.16b, v13.4b[1]\n"
                    ".inst 0x6fafe2bf // udot v31.4s, v21.16b, v15.4b[1]\n"
                    "ldr q21, [%[b_ptr0], #0x50]\n"
                    ".inst 0x6f81ead8 // udot v24.4s, v22.16b, v1.4b[2]\n"
                    ".inst 0x6f83ead9 // udot v25.4s, v22.16b, v3.4b[2]\n"
                    ".inst 0x6f85eada // udot v26.4s, v22.16b, v5.4b[2]\n"
                    ".inst 0x6f87eadb // udot v27.4s, v22.16b, v7.4b[2]\n"
                    ".inst 0x6f89eadc // udot v28.4s, v22.16b, v9.4b[2]\n"
                    ".inst 0x6f8beadd // udot v29.4s, v22.16b, v11.4b[2]\n"
                    ".inst 0x6f8deade // udot v30.4s, v22.16b, v13.4b[2]\n"
                    ".inst 0x6f8feadf // udot v31.4s, v22.16b, v15.4b[2]\n"
                    "ldr q22, [%[b_ptr0], #0x60]\n"
                    ".inst 0x6fa1eaf8 // udot v24.4s, v23.16b, v1.4b[3]\n"
                    ".inst 0x6fa3eaf9 // udot v25.4s, v23.16b, v3.4b[3]\n"
                    ".inst 0x6fa5eafa // udot v26.4s, v23.16b, v5.4b[3]\n"
                    ".inst 0x6fa7eafb // udot v27.4s, v23.16b, v7.4b[3]\n"
                    ".inst 0x6fa9eafc // udot v28.4s, v23.16b, v9.4b[3]\n"
                    ".inst 0x6fabeafd // udot v29.4s, v23.16b, v11.4b[3]\n"
                    ".inst 0x6fadeafe // udot v30.4s, v23.16b, v13.4b[3]\n"
                    ".inst 0x6fafeaff // udot v31.4s, v23.16b, v15.4b[3]\n"
                    "b.ne 8b\n"
                    "7:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "movi v24.4s, #0\n"
                    "ldr q23, [%[b_ptr0], #0x70]\n"
                    "add %[b_ptr0], %[b_ptr0], #0x80\n"
                    "str q25, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "movi v25.4s, #0\n"
                    ".inst 0x6f80e218 // udot v24.4s, v16.16b, v0.4b[0]\n"
                    "str q26, [c_ptr2]\n"
                    "movi v26.4s, #0\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    ".inst 0x6f82e219 // udot v25.4s, v16.16b, v2.4b[0]\n"
                    "str q27, [c_ptr3]\n"
                    "movi v27.4s, #0\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    ".inst 0x6f84e21a // udot v26.4s, v16.16b, v4.4b[0]\n"
                    "str q28, [c_ptr4]\n"
                    "movi v28.4s, #0\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    ".inst 0x6f86e21b // udot v27.4s, v16.16b, v6.4b[0]\n"
                    "str q29, [c_ptr5]\n"
                    "movi v29.4s, #0\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    ".inst 0x6f88e21c // udot v28.4s, v16.16b, v8.4b[0]\n"
                    "str q30, [c_ptr6]\n"
                    "movi v30.4s, #0\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    ".inst 0x6f8ae21d // udot v29.4s, v16.16b, v10.4b[0]\n"
                    "str q31, [c_ptr7]\n"
                    "movi v31.4s, #0\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".inst 0x6f8ce21e // udot v30.4s, v16.16b, v12.4b[0]\n"
                    ".inst 0x6fa0e238 // udot v24.4s, v17.16b, v0.4b[1]\n"
                    ".inst 0x6f8ee21f // udot v31.4s, v16.16b, v14.4b[0]\n"
                    ".inst 0x6fa2e239 // udot v25.4s, v17.16b, v2.4b[1]\n"
                    ".inst 0x6fa4e23a // udot v26.4s, v17.16b, v4.4b[1]\n"
                    ".inst 0x6fa6e23b // udot v27.4s, v17.16b, v6.4b[1]\n"
                    ".inst 0x6fa8e23c // udot v28.4s, v17.16b, v8.4b[1]\n"
                    ".inst 0x6faae23d // udot v29.4s, v17.16b, v10.4b[1]\n"
                    ".inst 0x6face23e // udot v30.4s, v17.16b, v12.4b[1]\n"
                    ".inst 0x6faee23f // udot v31.4s, v17.16b, v14.4b[1]\n"
                    ".inst 0x6f80ea58 // udot v24.4s, v18.16b, v0.4b[2]\n"
                    ".inst 0x6f82ea59 // udot v25.4s, v18.16b, v2.4b[2]\n"
                    ".inst 0x6f84ea5a // udot v26.4s, v18.16b, v4.4b[2]\n"
                    ".inst 0x6f86ea5b // udot v27.4s, v18.16b, v6.4b[2]\n"
                    ".inst 0x6f88ea5c // udot v28.4s, v18.16b, v8.4b[2]\n"
                    ".inst 0x6f8aea5d // udot v29.4s, v18.16b, v10.4b[2]\n"
                    ".inst 0x6f8cea5e // udot v30.4s, v18.16b, v12.4b[2]\n"
                    ".inst 0x6f8eea5f // udot v31.4s, v18.16b, v14.4b[2]\n"
                    ".inst 0x6fa0ea78 // udot v24.4s, v19.16b, v0.4b[3]\n"
                    ".inst 0x6fa2ea79 // udot v25.4s, v19.16b, v2.4b[3]\n"
                    ".inst 0x6fa4ea7a // udot v26.4s, v19.16b, v4.4b[3]\n"
                    ".inst 0x6fa6ea7b // udot v27.4s, v19.16b, v6.4b[3]\n"
                    ".inst 0x6fa8ea7c // udot v28.4s, v19.16b, v8.4b[3]\n"
                    ".inst 0x6faaea7d // udot v29.4s, v19.16b, v10.4b[3]\n"
                    ".inst 0x6facea7e // udot v30.4s, v19.16b, v12.4b[3]\n"
                    ".inst 0x6faeea7f // udot v31.4s, v19.16b, v14.4b[3]\n"
                    ".inst 0x6f81e298 // udot v24.4s, v20.16b, v1.4b[0]\n"
                    ".inst 0x6f83e299 // udot v25.4s, v20.16b, v3.4b[0]\n"
                    ".inst 0x6f85e29a // udot v26.4s, v20.16b, v5.4b[0]\n"
                    ".inst 0x6f87e29b // udot v27.4s, v20.16b, v7.4b[0]\n"
                    ".inst 0x6f89e29c // udot v28.4s, v20.16b, v9.4b[0]\n"
                    ".inst 0x6f8be29d // udot v29.4s, v20.16b, v11.4b[0]\n"
                    ".inst 0x6f8de29e // udot v30.4s, v20.16b, v13.4b[0]\n"
                    ".inst 0x6f8fe29f // udot v31.4s, v20.16b, v15.4b[0]\n"
                    ".inst 0x6fa1e2b8 // udot v24.4s, v21.16b, v1.4b[1]\n"
                    ".inst 0x6fa3e2b9 // udot v25.4s, v21.16b, v3.4b[1]\n"
                    ".inst 0x6fa5e2ba // udot v26.4s, v21.16b, v5.4b[1]\n"
                    ".inst 0x6fa7e2bb // udot v27.4s, v21.16b, v7.4b[1]\n"
                    ".inst 0x6fa9e2bc // udot v28.4s, v21.16b, v9.4b[1]\n"
                    ".inst 0x6fabe2bd // udot v29.4s, v21.16b, v11.4b[1]\n"
                    ".inst 0x6fade2be // udot v30.4s, v21.16b, v13.4b[1]\n"
                    ".inst 0x6fafe2bf // udot v31.4s, v21.16b, v15.4b[1]\n"
                    ".inst 0x6f81ead8 // udot v24.4s, v22.16b, v1.4b[2]\n"
                    ".inst 0x6f83ead9 // udot v25.4s, v22.16b, v3.4b[2]\n"
                    ".inst 0x6f85eada // udot v26.4s, v22.16b, v5.4b[2]\n"
                    ".inst 0x6f87eadb // udot v27.4s, v22.16b, v7.4b[2]\n"
                    ".inst 0x6f89eadc // udot v28.4s, v22.16b, v9.4b[2]\n"
                    ".inst 0x6f8beadd // udot v29.4s, v22.16b, v11.4b[2]\n"
                    ".inst 0x6f8deade // udot v30.4s, v22.16b, v13.4b[2]\n"
                    ".inst 0x6f8feadf // udot v31.4s, v22.16b, v15.4b[2]\n"
                    ".inst 0x6fa1eaf8 // udot v24.4s, v23.16b, v1.4b[3]\n"
                    ".inst 0x6fa3eaf9 // udot v25.4s, v23.16b, v3.4b[3]\n"
                    ".inst 0x6fa5eafa // udot v26.4s, v23.16b, v5.4b[3]\n"
                    ".inst 0x6fa7eafb // udot v27.4s, v23.16b, v7.4b[3]\n"
                    ".inst 0x6fa9eafc // udot v28.4s, v23.16b, v9.4b[3]\n"
                    ".inst 0x6fabeafd // udot v29.4s, v23.16b, v11.4b[3]\n"
                    ".inst 0x6fadeafe // udot v30.4s, v23.16b, v13.4b[3]\n"
                    ".inst 0x6fafeaff // udot v31.4s, v23.16b, v15.4b[3]\n"
                    "6:\n"
                    "str q24, [%[c_ptr0]]\n"
                    "add %[c_ptr0], %[c_ptr0], #0x10\n"
                    "str q25, [c_ptr1]\n"
                    "add c_ptr1, c_ptr1, #0x10\n"
                    "str q26, [c_ptr2]\n"
                    "add c_ptr2, c_ptr2, #0x10\n"
                    "str q27, [c_ptr3]\n"
                    "add c_ptr3, c_ptr3, #0x10\n"
                    "str q28, [c_ptr4]\n"
                    "add c_ptr4, c_ptr4, #0x10\n"
                    "str q29, [c_ptr5]\n"
                    "add c_ptr5, c_ptr5, #0x10\n"
                    "str q30, [c_ptr6]\n"
                    "add c_ptr6, c_ptr6, #0x10\n"
                    "str q31, [c_ptr7]\n"
                    "add c_ptr7, c_ptr7, #0x10\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "cc", "memory"
                );
                break;
        }
    }
}

} // namespace arm_gemm

#endif // __aarch64__
